#include <board.h> /* EVM library */
#include <limits.h>
#include <stdio.h>
#define _TI_ENHANCED_MATH_H 1
#include <math.h>
#include <img_histogram.h> /* IMGLIB */

/* image dimensions */
#define X_SIZE 128
#define Y_SIZE 128
#define N_PIXELS X_SIZE*Y_SIZE

unsigned char in_img[N_PIXELS];

/* Input & output buffer both will not fit in
   internal chip RAM, this pragma places the 
   output buffer in off-chip RAM */
#pragma DATA_SECTION (out_img, "SBSRAM"); 
unsigned char out_img[N_PIXELS];

/* image histogram goes here, the pragma
   aligns the buffer on a 4-byte boundary
   which is required by IMGLIB */
#pragma DATA_ALIGN (hist, 4)
unsigned short hist[256];

/* scratch buffer needed by IMGLIB */
unsigned short t_hist[1024];

void compute_range(unsigned char *pin, int N,
                   float percentile1, float percentile2,
                   unsigned char *pbin1, unsigned char *pbin2)
{
	unsigned short cumsum = 0, /* running tally of the cumulative sum */
	               T1 = round((float)N * percentile1), /* threshold for 1st bin */
	               T2 = round((float)N * percentile2); /* threshold for 2nd bin */
	int ii, jj;
	
	/* calc image histogram */
	
	/* buffers must be initialized to zero */
	memset(t_hist, 0, sizeof(unsigned short)*1024);
	memset(hist, 0, sizeof(unsigned short)*256);
	IMG_histogram(pin, N, 1, t_hist, hist);
	
	/* find location for 1st (lower bound) percentile in histogram */
	for (ii=0; ii<256; ++ii) {
		cumsum += hist[ii];
		if (cumsum >= T1) {
			*pbin1 = ii;
			break;
		}
	}
	
	/* find location for 2nd (upper bound) percentile in histogram */
	for (jj=ii+1; jj<256; ++jj) {
		cumsum += hist[jj];
		if (cumsum >= T2) {
			*pbin2 = jj;
			break;
		}
	}
}

void contrast_stretch(unsigned char *pin, 
                      unsigned char *pout, 
                      int N)
{
	unsigned char a, b; /* lower & upper bounds for scaling function */
	float scale, pixel;
	int ii;
	
	/* estimate dynamic range of the input */
	compute_range(pin, N, .05, 0.95, &a, &b);
	
	/* apply linear scaling function to input pixels,
	   taking care to handle overflow & underflow */
	scale = 255.0/(float)(b-a);
	for (ii=0; ii<N_PIXELS; ++ii) {
		pixel = round( (float)(pin[ii] - a) * scale );
		/* clamp to 8 bit range */
		pout[ii] = (pixel<0.f) ? 0 : ((pixel>255.0) ? 255 : pixel);
	}
} 

int main(void)
{
	evm_init(); /* initialize the board */
	contrast_stretch(in_img, out_img, N_PIXELS); 
	printf("contrast stretch completed");	
}
