#include <board.h> /* EVM library */
#include <stdio.h>
#define _TI_ENHANCED_MATH_H 1
#include <math.h>
#include <img_histogram.h> /* IMGLIB */
#include <time.h> /* for clock() */

/* image dimensions */
#define X_SIZE 128
#define Y_SIZE 128
#define N_PIXELS X_SIZE*Y_SIZE

unsigned char in_img[N_PIXELS];

/* Input & output buffer both will not fit in
   internal chip RAM, this pragma places the 
   output buffer in off-chip RAM */
#pragma DATA_SECTION (out_img, "SBSRAM"); 
unsigned char out_img[N_PIXELS];

/* image histogram goes here, the pragma
   aligns the buffer on a 4-byte boundary
   which is required by IMGLIB */
#pragma DATA_ALIGN (hist, 4)
unsigned short hist[256];

/* scratch buffer needed by IMGLIB */
#pragma DATA_ALIGN (t_hist, 4)
unsigned short t_hist[1024];

/* pixel mapping function */
#pragma DATA_ALIGN (T_r, 4)
unsigned short T_r[256];

/* timing */
clock_t start, stop, overhead, t;

#define FIXED_POINT 0 /* 0=floating point, 1=fixed point */
#if FIXED_POINT

unsigned int cumsum; // 32-bit acummulator

#else // floating-point

const float num_pixels = (float)N_PIXELS;
float cumsum;

#endif

clock_t histeq()
{
	int ii = 0;
	cumsum = 0;

	/* buffers must be initialized to zero */
	memset(t_hist, 0, sizeof(unsigned short)*1024);
	memset(hist, 0, sizeof(unsigned short)*256);
	
	/* use IMGLIB to compute image histogram */
	IMG_histogram(in_img, N_PIXELS, 1, t_hist, hist);
	
	/* form gray-level tranform function T(r) = CDF */
#if FIXED_POINT
	start = clock(); /* begin "profile area" */
	for (ii=0; ii<256; ++ii) {
		cumsum += (hist[ii] << 2);
		T_r[ii] = (unsigned short) (255*cumsum) >> 16;
	}
	stop = clock(); /* end "profile area" */
	t = stop-start-overhead;
	printf("Fixed point compute T_r cycles: %d\n", t);
	
#else

	start = clock(); /* begin "profile area" */
	for (ii=0; ii<256; ++ii) {
		cumsum += hist[ii];
		T_r[ii] = round(255.f*cumsum/num_pixels);
	}
	stop = clock(); /* end "profile area" */
	t = stop-start-overhead;
	printf("Floating point compute T_r cycles: %d\n", t);

#endif

	/* apply histogram equalization transform function */
	for (ii=0 ; ii<N_PIXELS; ++ii)
		out_img[ii] = T_r[in_img[ii]];
		
	return t;
}

int main(void)
{
	const int N = 10; /* how many times to profile */
	int ii = 0;
	clock_t profile = 0;
	
	evm_init(); /* initialize the board */
	start = clock(); /* calculate overhead of calling clock*/
	stop = clock();  /* and subtract this value from The results*/
	overhead = stop - start;
	
	for (; ii<N; ++ii)
		profile += histeq();
	
	printf("average time to compute T_r is %.2f cycles\n", (float)profile/(float)N);	
}
