#include "stdafx.h"

using namespace std;
using namespace Gdiplus;

//
// Constructor: which instantiates GDI+ object for display that uses 
// as a IPP-compatible array for pixel storage.
//
Image8bpp::Image8bpp(const CString &filename) : m_pBitmap(NULL), m_pPixels(NULL)
{
	// ASCII -> UNICODE conversion
	int len = filename.GetLength();
	vector<wchar_t> wName(len+1);
	mbstowcs (&wName[0], (LPCSTR)filename, len);
	wName[len] = L'\0';

	// use this temporary object purely to deal with
	// the machinations of performing the file I/O
	Status s = Ok;
	Bitmap bmp(&wName[0]);
	if (Ok != (s = bmp.GetLastStatus()))
		throw std::runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// now allocate aligned memory for use with Intel IPP functions
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(bmp.GetWidth(), bmp.GetHeight(), &m_stride)))
		throw std::runtime_error("Out of memory.");

	BitmapData bmpData;
	Rect rect(0, 0, bmp.GetWidth(), bmp.GetHeight());
	PixelFormat fmt = bmp.GetPixelFormat();
	if (PixelFormat24bppRGB == fmt) { // must convert from RGB to grayscale

		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat24bppRGB, &bmpData);
		if (Ok != s)
			throw std::runtime_error( (LPCSTR)GdiplusUtil::getErrorString(s) );

		// color conversion (note that even though we'd like to use the IPP
		// function ippiRGBToGray we can't because Microsoft stores pixel
		// values in BGR format)
		unsigned char *pInput = (unsigned char*)bmpData.Scan0, *pScanIn = NULL;
		Ipp8u *pOutput = m_pPixels, *pScanOut = NULL;
		for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow) {
			pScanIn = pInput + iRow*bmpData.Stride;
			pScanOut = pOutput + iRow*m_stride;
			for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol)
				pScanOut[iCol] = (Ipp8u)(pScanIn[iCol*3]*0.114 + pScanIn[iCol*3+1]*0.587 + pScanIn[iCol*3+2]*0.299);
		}

	} else if (PixelFormat8bppIndexed != fmt) { // color conversion not necessary

		// get to the raw bits comprising the temporary GDI+ bitmap object
		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat8bppIndexed, &bmpData);
		if (Ok != s)
			throw std::runtime_error( (LPCSTR)GdiplusUtil::getErrorString(s) );

		// copy from temporary GDI+ object into aligned memory buffer
		unsigned char *pInput = (unsigned char*)bmpData.Scan0, *pScanIn = NULL;
		Ipp8u *pOutput = m_pPixels, *pScanOut = NULL;
		for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow) {
			pScanIn = pInput + iRow*bmpData.Stride;
			pScanOut = pOutput + iRow*m_stride;
			for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol)
				pScanOut[iCol] = pScanIn[iCol];
		}
	
	} else
		throw std::runtime_error("Only 8bpp indexed or 24bpp RGB images supported.");

	// finally, instantiate the GDI+ object that will by used for display purposes
	// (note that Bitmap ctor will not perform a deep copy, m_pPixels is shared) 
	m_pBitmap = new Bitmap(bmp.GetWidth(), bmp.GetHeight(), m_stride, PixelFormat8bppIndexed, m_pPixels);
	if (Ok != (s = m_pBitmap->GetLastStatus()))
		throw std::runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}

//
// Instantiate an image consisting of all zero pixels
//
Image8bpp::Image8bpp(int nr, int nc) : m_pBitmap(NULL), m_pPixels(NULL)
{
	// allocate aligned memory
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(nc, nr, &m_stride)))
		throw std::runtime_error("Out of memory.");

	IppiSize roi = {nc, nr};
	if (ippStsNoErr != ippiSet_8u_C1R(0, m_pPixels, m_stride, roi))
		throw std::runtime_error("ippiSet_8u_C1R() error");

	// instantiate the GDI+ object that will by used for display purposes
	// (note that Bitmap ctor will not perform a deep copy, m_pPixels is shared) 
	m_pBitmap = new Bitmap(nr, nc, m_stride, PixelFormat8bppIndexed, m_pPixels);
	Status s = m_pBitmap->GetLastStatus();
	if (Ok != s)
		throw std::runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}

//
// Copy constructor
//
Image8bpp::Image8bpp(const Image8bpp &other)
{
	if (other.m_pBitmap && other.m_pPixels)
		this->copy(other);
}

//
// Assignment operator
//
Image8bpp &Image8bpp::operator =(const Image8bpp &rhs)
{
	if (this != &rhs) {

		if (m_pBitmap) 
			delete m_pBitmap;

		if (m_pPixels)
			ippiFree(m_pPixels);

		this->copy(rhs);
	}

	return *this;
}

//
// Destructor: clean up objects
//
Image8bpp::~Image8bpp()
{
	if (m_pBitmap) 
	{
		delete m_pBitmap;
		m_pBitmap = 0;
	}

	if (m_pPixels)
	{
		ippiFree(m_pPixels);
		m_pPixels = 0;
	}
}

//
// This method will draw the bitmap to the specified control
//
void Image8bpp::render(HWND hwnd)
{
	if (!m_pBitmap)
		throw std::runtime_error(_T("bitmap pointer is NULL"));

	Graphics gr(hwnd); // instantiate GDI+ graphics object

	// grab destination region to BITBLT to
	RECT rc;
	::GetClientRect(hwnd, &rc);

	Rect destRect(rc.left, rc.top, 
		          (rc.right - rc.left),
				  rc.bottom - rc.top);

    Status s;
	if (Ok != (s = gr.DrawImage(m_pBitmap, destRect)))
		throw std::runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));
}

//
// Assumes that m_pPixels and m_pBitmap are either NULL or have already been deallocated
//
void Image8bpp::copy(const Image8bpp &src)
{
	m_stride = src.m_stride;

	// allocate buffer for pixel data
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight(), &m_stride)))
		throw std::runtime_error("Out of memory.");

	// copy
	IppiSize roiSize = {src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight()};
	if (ippStsNoErr != ippiCopy_8u_C1R(src.m_pPixels, src.m_stride, m_pPixels, m_stride, roiSize))
		throw std::runtime_error("Copy operation failed.");

	// instantiate GDI+ object
	m_pBitmap = new Bitmap(src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight(), m_stride, PixelFormat8bppIndexed, m_pPixels);
	Status s;
	if (Ok != (s = m_pBitmap->GetLastStatus()))
		throw std::runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}