function J = median_filter_image(I, sz)
%
% MEDIAN_FILTER_IMAGE   pass image through NxN median filter
%    J = MEDIAN_FILTER_IMAGE(I, sz)
%
%    J: filtered image
%    I: input image, must be UINT8, UINT16, or double
%    sz: size of the median filter, must be odd
%

if ~mod(sz(1), 2) | ~mod(sz(2), 2)
    error('neighborhood dimensions must be odd!');
end
imedian = ceil(prod(sz)/2);

[MI, NI] = size(I);
J = zeros(MI, NI);

ihw = floor(sz(1)/2); % ihw = i "half-width"
jhw = floor(sz(2)/2); % jhw = j "half-width"

% this double-loop slides the neighborhood around the image
for ii = ihw+1:MI-ihw
    for jj = jhw+1:NI-jhw
        % splice out the portion of the image
        neighborhood = I(ii-ihw:ii+ihw, jj-jhw:jj+jhw);
        % sort the neighborhood (in ascending order)
        n_sorted = sort(neighborhood(:));
        % generate output pixel
        J(ii, jj) = n_sorted(imedian);
    end
end

if isa(I, 'uint8')
    J = uint8(J);
elseif isa(I, 'uint16')
    J = uint16(J);
end