function [J, varargout] = hist_specification(I, hgram)
%
% HIST_SPECIFICATION   histogram specification
%    J = HIST_SPECIFICATION(I, hgram)
%    [J, LUT] = HIST_SPECIFICATION(I, hgram)
%
%    J: processed image, whose histogram will match
%       that given in hgram
%    LUT: pixel remapping function
%    I: input image, may be UINT8 or UINT16
%    hgram: desired histogram
%

if strcmp('uint8', class(I))
    nlevels = 256;
elseif strcmp('uint16', class(I))
    nlevels = 65536;
else
    error('Require 8-bit or 16-bit input image');
end

% this function expects a normalized target histogram
if max(hgram(:) > 1)
    error('Please divide target hist by # pixels in desired image!');
end

% calculate normalized histogram of input image
H = histc(I(:), 0:nlevels-1);
H = H ./ length(I(:));

% cumulative histograms of both input and desired distributions
cdf_orig = cumsum(H);
cdf_desired = cumsum(hgram);

% construct gray-level transformation lookup table here:
% scan along input CDF, stopping when cumulative histogram
% exceeds target CDF.
p = 0;
for ii=0:255
    while cdf_desired(p+1)<cdf_orig(ii+1) && p<255
        p = p + 1;
    end
    LUT(ii+1) = p;
end


% apply LUT to input image I
% NOTE: must promote to real because certain operations
%       not allowed with integer objects, and add 1 because
%       MATLAB is one-based yet pixel intensities are 0-based.
J = LUT(double(I) + 1);

% if client asks for it, return the LUT
if 2==nargout
    varargout{1} = LUT;
end