function J = add_shot_noise(I, how_much)
%
% ADD_SHOT_NOISE   corrupt an image with "salt and pepper" noise
%    J = ADD_SHOT_NOISE(I, how_much)
%
%    J: noisy image
%    I: input image, may be UINT8, UINT16, or double
%    how_much: noise density
%
% NOTE: if I is of type double, then pixel values should lie within
%       the range [0,1]
%

if how_much > 1.0
    error('Cannot corrupt more than 100% of the pixels');
end
[M N] = size(I);

% rand returns a MxN matrix with uniformly distributed
% #'s in the interval (0.0, 1.0), so by finding how
% many of these are less than or equal to how_much we
% determine which pixels are to be corrupted. 
R = rand(M, N);
icorrupted = find(R <= how_much);
ncorrupted = length(icorrupted);

% generate a normally distributed vector with zero mean,
% so half of them should be negative, and half of them
% positive.
salt_and_pepper = randn(1, ncorrupted);

% the signum function maps negative #'s to -1 and
% positive #'s to +1.
salt_and_pepper = sign(salt_and_pepper);

% noisy is a temporary matrix to store the pixel
% indices of the corrupted data
noisy = zeros(M, N);
noisy(icorrupted) = salt_and_pepper;

% those locations in noisy that are -1 will be set
% to the 'off' (min) intensity, and those that are
% +1 will be set to the 'on' (max) intensity.
min_noise_val = 0;
if isa(I, 'uint8')
    max_noise_val = 255;
elseif isa(I, 'uint16')
    max_noise_val = 65535;
else
    max_noise_val = 1;
end

J = I;
icorrupted = find(-1 == noisy);
J(icorrupted) = min_noise_val;
icorrupted = find(1 == noisy);
J(icorrupted) = max_noise_val;