#include "stdafx.h"

using namespace std;
using namespace Gdiplus;

//
// Constructor: which instantiates GDI+ object for display that uses 
// as a IPP-compatible array for pixel storage.
//
Image8bpp::Image8bpp(const CString &filename) : m_pBitmap(NULL), m_pPixels(NULL), m_stride(0)
{
	// ASCII -> UNICODE conversion
	int len = filename.GetLength();
	vector<wchar_t> wName(len+1);
	mbstowcs (&wName[0], (LPCSTR)filename, len);
	wName[len] = L'\0';

	// use this temporary object purely to deal with
	// the machinations of performing the file I/O
	Status s = Ok;
	Bitmap bmp(&wName[0]);
	if (Ok != (s = bmp.GetLastStatus()))
		throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// allocate aligned memory for use with Intel IPP functions
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(bmp.GetWidth(), bmp.GetHeight(), &m_stride)))
		throw runtime_error("Out of memory.");

	BitmapData bmpData;
	Rect rect(0, 0, bmp.GetWidth(), bmp.GetHeight());
	PixelFormat fmt = bmp.GetPixelFormat();

	if (PixelFormat24bppRGB == fmt) { // must convert from RGB to grayscale

		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat24bppRGB, &bmpData);
		if (Ok != s)
			throw runtime_error( (LPCSTR)GdiplusUtil::getErrorString(s) );

		// color conversion (note that even though we'd like to use the IPP
		// function ippiRGBToGray we can't because Microsoft stores pixel
		// values in BGR format)
		unsigned char *pInput = (unsigned char*)bmpData.Scan0, *pScanIn = NULL;
		Ipp8u *pOutput = m_pPixels, *pScanOut = NULL;
		for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow) {
			pScanIn = pInput + iRow*bmpData.Stride;
			pScanOut = pOutput + iRow*m_stride;
			for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol)
				pScanOut[iCol] = (Ipp8u)(pScanIn[iCol*3]*0.114 + pScanIn[iCol*3+1]*0.587 + pScanIn[iCol*3+2]*0.299);
		}

	} else if (PixelFormat8bppIndexed != fmt) { // color conversion not necessary

		// get to the raw bits comprising the temporary GDI+ bitmap object
		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat8bppIndexed, &bmpData);
		if (Ok != s)
			throw runtime_error( (LPCSTR)GdiplusUtil::getErrorString(s) );

		// copy from temporary GDI+ object into aligned memory buffer
		unsigned char *pInput = (unsigned char*)bmpData.Scan0, *pScanIn = NULL;
		Ipp8u *pOutput = m_pPixels, *pScanOut = NULL;
		for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow) {
			pScanIn = pInput + iRow*bmpData.Stride;
			pScanOut = pOutput + iRow*m_stride;
			for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol)
				pScanOut[iCol] = pScanIn[iCol];
		}
	
	} else
		throw runtime_error("Only 8bpp indexed or 24bpp RGB images supported.");

	// finally, instantiate the GDI+ object that will by used for display purposes
	// (note that Bitmap ctor will not perform a deep copy, m_pPixels is shared) 
	m_pBitmap = new Bitmap(bmp.GetWidth(), bmp.GetHeight(), m_stride, PixelFormat8bppIndexed, m_pPixels);
	if (Ok != (s = m_pBitmap->GetLastStatus()))
		throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}

//
// Instantiate an image consisting of all zero pixels if pSrc is NULL,
// or otherwise copy pixel data into new object.
//
Image8bpp::Image8bpp(int nr, int nc, int srcStride /* = -1 */, Ipp8u *pSrc /* = NULL */)
: m_pBitmap(NULL), m_pPixels(NULL)
{
	// allocate aligned memory
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(nc, nr, &m_stride)))
		throw runtime_error("Out of memory.");

	IppiSize roi = {nc, nr};
	if (!pSrc) { // set entire image to 0
		if (ippStsNoErr != ippiSet_8u_C1R(0, m_pPixels, m_stride, roi))
			throw runtime_error("ippiSet_8u_C1R() error");
	} else { // copy pixel data
		if (ippStsNoErr != ippiCopy_8u_C1R(pSrc, srcStride, m_pPixels, m_stride, roi))
			throw runtime_error("ippiCopy_8u_C1R() error");
	}

	// instantiate the GDI+ object that will by used for display purposes
	// (note that Bitmap ctor will not perform a deep copy, m_pPixels is shared) 
	m_pBitmap = new Bitmap(nr, nc, m_stride, PixelFormat8bppIndexed, m_pPixels);
	Status s = m_pBitmap->GetLastStatus();
	if (Ok != s)
		throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}

//
// Copy constructor
//
Image8bpp::Image8bpp(const Image8bpp &other)
{
	if (other.m_pBitmap && other.m_pPixels)
		this->copy(other);
}

//
// Assignment operator
//
Image8bpp &Image8bpp::operator =(const Image8bpp &rhs)
{
	if (this != &rhs) {

		if (m_pBitmap) 
			delete m_pBitmap;

		if (m_pPixels)
			ippiFree(m_pPixels);

		this->copy(rhs);
	}

	return *this;
}

//
// Destructor: clean up objects
//
Image8bpp::~Image8bpp()
{
	if (m_pBitmap) 
	{
		delete m_pBitmap;
		m_pBitmap = 0;
	}

	if (m_pPixels)
	{
		ippiFree(m_pPixels);
		m_pPixels = 0;
	}
}

//
// This method will draw the bitmap to the specified control
//
void Image8bpp::render(HWND hwnd)
{
	if (!m_pBitmap)
		throw runtime_error(_T("bitmap pointer is NULL"));

	Graphics gr(hwnd); // instantiate GDI+ graphics object

	// grab destination region to BITBLT to
	RECT rc;
	::GetClientRect(hwnd, &rc);

	Rect destRect(rc.left, rc.top, 
		          (rc.right - rc.left),
				  rc.bottom - rc.top);

    Status s;
	if (Ok != (s = gr.DrawImage(m_pBitmap, destRect)))
		throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));
}

//
// More or less a port of MATLAB's imresize function
//
Image8bpp &Image8bpp::resize(int newWidth, int newHeight)
{
	if (m_pBitmap && m_pPixels) {

		Ipp8u *pResizedPixels = NULL;
		Bitmap *pResizedBitmap = NULL;
		try {

			IppiSize srcSize = {m_pBitmap->GetWidth(), m_pBitmap->GetHeight()};
			IppiRect srcRoi  = {0, 0, srcSize.width, srcSize.height};
			IppiSize dstRoiSize = {newWidth, newHeight};

			// allocate new buffers, that will replace current ones
			int newStride;
			if (NULL == (pResizedPixels = ippiMalloc_8u_C1(newWidth, newHeight, &newStride)))
				throw runtime_error("Out of memory.");

			// resize the image pixel data
			double xFactor = (double)newWidth/(double)srcSize.width,
				   yFactor = (double)newHeight/(double)srcSize.height;
			if (ippStsNoErr != ippiResize_8u_C1R(m_pPixels, srcSize, m_stride, srcRoi,
												pResizedPixels, newStride, dstRoiSize,
												xFactor, yFactor,
												IPPI_INTER_CUBIC)) {
		
				throw runtime_error("Resize operation failed.");

			}

			// instantiate new Bitmap object
			pResizedBitmap = new Bitmap(newHeight, newWidth, newStride, PixelFormat8bppIndexed, pResizedPixels);
			Status s = pResizedBitmap->GetLastStatus();
			if (Ok != s)
				throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

			// without the correct color palette the gray-scale image will not display correctly
			pResizedBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());

			// update this object's internal data structures
			m_stride = newStride;
			ippiFree(m_pPixels); m_pPixels = pResizedPixels;
			delete m_pBitmap; m_pBitmap = pResizedBitmap;

		} catch (runtime_error &e) {

			if (pResizedBitmap)
				delete pResizedBitmap;
			if (pResizedPixels)
				ippiFree(pResizedPixels);
			throw (e);

		}

	} // end (if this image contains pixel data)

	return *this;
}

//
// Treats all non-zero pixels as logical value true, or 255 for display purposes
//
void Image8bpp::binarize()
{
	IppiSize size = {m_pBitmap->GetWidth(), m_pBitmap->GetHeight()};
	if (m_pPixels)
		if (ippStsNoErr != ippiThreshold_Val_8u_C1IR(m_pPixels, m_stride, size, 0, 255, ippCmpGreater))
			throw runtime_error("ippiThreshold_Val_8u_C1R failed");
}

//
// Assumes that m_pPixels and m_pBitmap are either NULL or have already been deallocated
//
void Image8bpp::copy(const Image8bpp &src)
{
	m_stride = src.m_stride;

	// allocate buffer for pixel data
	if (NULL == (m_pPixels = ippiMalloc_8u_C1(src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight(), &m_stride)))
		throw runtime_error("Out of memory.");

	// copy
	IppiSize roiSize = {src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight()};
	if (ippStsNoErr != ippiCopy_8u_C1R(src.m_pPixels, src.m_stride, m_pPixels, m_stride, roiSize))
		throw runtime_error("Copy operation failed.");

	// instantiate GDI+ object
	m_pBitmap = new Bitmap(src.m_pBitmap->GetWidth(), src.m_pBitmap->GetHeight(), m_stride, PixelFormat8bppIndexed, m_pPixels);
	Status s;
	if (Ok != (s = m_pBitmap->GetLastStatus()))
		throw runtime_error((LPCSTR)GdiplusUtil::getErrorString(s));

	// without the correct color palette the gray-scale image will not display correctly
	m_pBitmap->SetPalette(GdiplusUtil::get8bppGrayScalePalette());
}