#include <board.h> /* EVM library */
#include "wave_edge.h"

#pragma DATA_SECTION (out_img, "SBSRAM");
#pragma DATA_SECTION (in_img, "SBSRAM");

int wave_decomp_a_trous(int nlevels)
{
  int ilevel, irow, jcol, kk;
  for (ilevel=0; ilevel<nlevels; ++ilevel) {
    int iapprox = ilevel+1;

    /* 
     * zero out the margins of the current 
     * approximation coefficient matrix -
     * 1st and last couple of rows
     */
    for (kk=0; kk<Y_SIZE; ++kk)
      approx[0][kk][iapprox] = 
      approx[1][kk][iapprox] = 
      approx[X_SIZE-1][kk][iapprox] = 
      approx[X_SIZE-2][kk][iapprox] = 0;
    
    /* first and last couple of columns */
    approx[2][0][iapprox] = approx[2][1][iapprox] = 0;
    approx[2][Y_SIZE-2][iapprox] = approx[2][Y_SIZE-1][iapprox] = 0; 
    approx[3][0][iapprox] = approx[3][1][iapprox] = 0;
    approx[3][Y_SIZE-2][iapprox] = approx[3][Y_SIZE-1][iapprox] = 0;    
    
    /* 5x5 convolution for approximation coeffs at level i */
    for (irow=2; irow<(X_SIZE-2); ++irow) {
    
      approx[irow+2][0][iapprox] = approx[irow+2][1][iapprox] = 0;
      approx[irow+2][Y_SIZE-2][iapprox] = 0; 
      approx[irow+2][Y_SIZE-1][iapprox] = 0;
       
      for (jcol=2; jcol<(Y_SIZE-2); ++jcol) {

                                       /* row 1: 1/256 1/64 3/128 1/64 1/256 */
         approx[irow][jcol][iapprox] = (approx[irow-2][jcol-2][ilevel]>>8) +
                                       (approx[irow-2][jcol-1][ilevel]>>6) +
                                       ((3*approx[irow-2][jcol][ilevel])>>7) +
                                       (approx[irow-2][jcol+1][ilevel]>>6) +
                                       (approx[irow-2][jcol+2][ilevel]>>8) +
                                       /* row 2: 1/64 1/16 3/32 1/16 1/64 */
                                       (approx[irow-1][jcol-2][ilevel]>>6) +
                                       (approx[irow-1][jcol-1][ilevel]>>4) +
                                       ((3*approx[irow-1][jcol][ilevel])>>5) +
                                       (approx[irow-1][jcol+1][ilevel]>>4) +
                                       (approx[irow-1][jcol+2][ilevel]>>6) +
                                       /* row 3: 3/128 3/32 9/64 3/32 3/128 */
                                       ((3*approx[irow][jcol-2][ilevel])>>7) +
                                       ((3*approx[irow][jcol-1][ilevel])>>5) +
                                       ((9*approx[irow][jcol][ilevel])>>6) +
                                       ((3*approx[irow][jcol+1][ilevel])>>5) +
                                       ((3*approx[irow][jcol+2][ilevel])>>7) +
                                       /* row 4: 1/64 1/16 3/32 1/16 1/64 */
                                       (approx[irow+1][jcol-2][ilevel]>>6) +
                                       (approx[irow+1][jcol-1][ilevel]>>4) +
                                       ((3*approx[irow+1][jcol][ilevel])>>5) +
                                       (approx[irow+1][jcol+1][ilevel]>>4) +
                                       (approx[irow+1][jcol+2][ilevel]>>6) +
                                       /* row 5: 1/256 1/64 3/128 1/64 1/256 */ 
                                       (approx[irow+2][jcol-2][ilevel]>>8) +
                                       (approx[irow+2][jcol-1][ilevel]>>6) +
                                       ((3*approx[irow+2][jcol][ilevel])>>7) +
                                       (approx[irow+2][jcol+1][ilevel]>>6) +
                                       (approx[irow+2][jcol+2][ilevel]>>8); 
                                                                            
      }
    }
    /* 
     * end 5x5 convolution, now calc detail coeffs at level i
     * (but things are a bit different the final time through this loop)
     */
    if (ilevel != nlevels-1) { /* not the last time */
    
      for (irow=2; irow<(X_SIZE-2); ++irow)
        for (jcol=2; jcol<(Y_SIZE-2); ++jcol)
          detail[irow][jcol][ilevel] = approx[irow][jcol][iapprox] - approx[irow][jcol][ilevel];
          
    } else { /* last time, prep for segmentation of edge image */
    
      unsigned int sum = 0;
      for (irow=2; irow<(X_SIZE-2); ++irow) {
        for (jcol=2; jcol<(Y_SIZE-2); ++jcol) {
          detail[irow][jcol][ilevel] = 
            (_abs(approx[irow][jcol][iapprox] - approx[irow][jcol][ilevel])) >> 9;
          sum += detail[irow][jcol][ilevel];
        }
      }
      /* 
       * calculate average detail coefficient value for the 
       * highest decomposition level 
       */
      return (sum>>LOG2_N_PIXELS); /* div by (X_SIZE*Y_SIZE) */
      
    }
  } /* end (for each decomposition level) */
}

void segment_detail_image(int ilevel, int mean)
{
  int p;
  const int M = 3640; /* 1/9 in Q.15 format */
  unsigned long avg;
  int irow, jcol;
  ilevel -= 1;
  /* 
   * mark those pixels where the detail coefficient
   * is greater than the local mean AND where it is greater
   * than the global mean as an edge.
   */
  for (irow=3; irow<(X_SIZE-3); ++irow) {
    for (jcol=3; jcol<(Y_SIZE-3); ++jcol) {
      p = detail[irow][jcol][ilevel];
      /* find local average using 3x3 averaging filter */ 
      avg = ( detail[irow-1][jcol-1][ilevel]*M +
              detail[irow-1][jcol][ilevel]*M +
              detail[irow-1][jcol+1][ilevel]*M +
              detail[irow][jcol-1][ilevel]*M +
              p*M +
              detail[irow][jcol+1][ilevel]*M +
              detail[irow+1][jcol-1][ilevel]*M +
              detail[irow+1][jcol][ilevel]*M +
              detail[irow+1][jcol+1][ilevel]*M ) >> 15;
      out_img[irow*Y_SIZE+jcol] = (p>avg && p>mean) ? 255 : 0;
    }
  }
}

int main(void)
{
  const int nlevels = 3;
  int irow=0, jcol=0;
  int mean;
  
  evm_init(); /* initialize the board */

  for (irow=0; irow<X_SIZE; ++irow)
    for (jcol=0; jcol<Y_SIZE; ++jcol)
      /* scale input for fixed-point arithmetic */
	  approx[irow][jcol][0] = (in_img[irow*Y_SIZE+jcol] << 9);  
      
  mean = wave_decomp_a_trous(nlevels);
  segment_detail_image(nlevels, mean);
}
