-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Analog Voltage Controlled Oscillator
-- 
-- Description:
-- The model describes a VCO. The frequency of the output voltage
-- source is controlled by the input voltage. The frequency equals 
-- KF*VIN + F0.
-- 
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- vco.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
use IEEE.MATH_REAL.all;


entity VCO is
   generic (F0            : REAL := 1.0; -- center frequency [Hz]
            KF            : REAL := 1.0; -- gain [Hz/V]
            AMPL          : REAL := 1.0; -- amplitude [V]
            PHI0          : REAL := 0.0  -- initial phase [rad]
           );
   port    (terminal INP  : ELECTRICAL;  -- input terminal
            terminal OUTP : ELECTRICAL   -- output terminal
           );
begin
   assert F0 > 0.0 and KF > 0.0
     report "F0 and KF > 0.0 required."
     severity ERROR;
end entity VCO;


architecture BASIC of VCO is
  quantity VIN  across INP;               -- open input branch
  quantity VOUT across IOUT through OUTP; -- output branch
  quantity PHI  : REAL;                   -- free quantity PHI
begin
  if DOMAIN = QUIESCENT_DOMAIN use
     PHI == PHI0;                         -- initial condition
  else
     PHI'DOT == MATH_2_PI*(KF*VIN + F0);  -- see VCO description
  end use;
  VOUT == AMPL*SIN(PHI);                  -- see VCO description
end architecture BASIC;
