-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      A-to-D Converter
-- 
-- Description:
-- The output signal is '1' if the input voltage is above LEVEL.
-- Otherwise it is '0'.
-- The architecture EXTENDED includes a hysteresis HYST.
-- Thus, changes to '1' occur at LEVEL+HYST/2. Changes to '0'
-- occur at LEVEL-HYST/2.
-- 
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    STD_LOGIC_1164
-- -----------------------------------------------------------
--
-- Source:
-- a2d.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
    use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
    use IEEE.STD_LOGIC_1164.all;
   
entity A2D is
  generic (LEVEL : REAL := 2.5;   -- threshold level [V]
           HYST  : REAL := 0.0    -- hysteresis [V], MIN: >= 0.0
          );
  port    (terminal INP   : ELECTRICAL;          -- controlling terminal
           signal   S_OUT : out STD_LOGIC := '0' -- output signal
          );
begin
  assert HYST >= 0.0
    report "ERROR: Hysteresis HYST >= 0.0 required."
    severity ERROR;
end entity A2D;

architecture IDEAL of A2D is
  quantity VIN across INP; 
begin
  S_OUT <= '1' when VIN'ABOVE(LEVEL) else '0';
end architecture IDEAL;

architecture EXTENDED of A2D is
  quantity VIN across INP; 
begin
  S_OUT <= '1' when VIN'ABOVE(LEVEL + HYST/2.0) else
           '0' when not VIN'ABOVE(LEVEL - HYST/2.0);
end architecture EXTENDED;
