-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:     Sinusoidal source with power specification
-- 
-- Description:
-- The model describes independent and modulated sources
-- where the amplitude is specified in terms of power.
-- There are architectures with single- and two-tone
-- as well as amplitude and frequency modulted sine waves
-- available.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- p_sin.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity P_SIN is 
  generic (VO      : REAL := 0.0;    -- offset voltage [V]
           PA_DBM  : REAL := -100.0; -- first tone amplitude [dBm]
           FREQ    : REAL := 1.0e03; -- first tone frequency [Hz]
	   PHASE   : REAL := 0.0;    -- first tone phase [rad]
           PA2_DBM : REAL := -100.0; -- second tone amplitude [dBm]
           FREQ2   : REAL := 1.0e03; -- second tone frequency [Hz]
	   PHASE2  : REAL := 0.0;    -- second tone phase [rad]
	   FREQM   : REAL := 1.0;    -- modulating tone frequency [Hz]
	   PHASEM  : REAL := 0.0;    -- modulating tone phase [rad]
	   MDI     : REAL := 0.0;    -- modulation index [-]
           ROUT    : REAL := 50.0    -- output impedance [Ohm]
	   );
  port (terminal P : ELECTRICAL;     -- positive terminal
        terminal M : ELECTRICAL      -- negative terminal
	);
end entity P_SIN;   

architecture SINGLE_TONE of P_SIN is
  constant PA: real:= 10**((PA_DBM-30.0)/10.0); 
  constant VA: real:= SQRT(PA * 2.0 * ROUT); 
  terminal N_INT: ELECTRICAL;
  quantity V_ROUT across I_ROUT through P to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to M;
begin

    -- signal source
    V_SRC == 2.0 * (VO + VA * SIN(NOW * MATH_2_PI*FREQ + PHASE));

    -- output port resistance
    V_ROUT == ROUT * I_ROUT;

end architecture SINGLE_TONE;                                            

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

architecture TWO_TONE of P_SIN is
  constant PA: REAL:= 10**((PA_DBM-30.0)/10.0); 
  constant VA: REAL:= SQRT(PA * 2.0 * ROUT); 
  constant PA2: REAL:= 10**((PA2_DBM-30.0)/10.0); 
  constant VA2: REAL:= SQRT(PA2 * 2.0 * ROUT); 
  terminal N_INT: ELECTRICAL;
  quantity V_ROUT across I_ROUT through P to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to M;
begin

  -- signal source
  V_SRC == 2.0 * (VO + VA * SIN(NOW * MATH_2_PI*FREQ + PHASE) +
      	              VA2 * SIN(NOW * MATH_2_PI*FREQ2 + PHASE2));
  -- output port resistance
  V_ROUT == ROUT * I_ROUT;

end architecture two_tone;

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

architecture AM of P_SIN is
  constant PA: REAL:= 10**((PA_DBM-30.0)/10.0); 
  constant VA: REAL:= SQRT(PA * 2.0 * ROUT); 
  terminal N_INT: ELECTRICAL;
  quantity V_ROUT across I_ROUT through P to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to M;
begin

  -- signal source
  V_SRC == 2.0 * (VO + VA * (1.0+MDI*SIN(NOW * MATH_2_PI*FREQM + PHASEM))
      	                   * SIN(NOW * MATH_2_PI*FREQ + PHASE));
  -- output port resistance
  V_ROUT == ROUT * I_ROUT;

end architecture am;                                            

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

architecture SFFM of P_SIN is
  constant PA: REAL:= 10**((PA_DBM-30.0)/10.0); 
  constant VA: REAL:= SQRT(PA * 2.0 * ROUT); 
  terminal N_INT: ELECTRICAL;
  quantity V_ROUT across I_ROUT through P to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to M;
begin

  -- signal source
  V_SRC == 2.0 * (VO + VA * SIN(NOW * MATH_2_PI*FREQ + PHASE
      	              + MDI*SIN(NOW * MATH_2_PI*FREQM + PHASEM)));
  -- output port resistance
  V_ROUT == ROUT * I_ROUT;

end architecture SFFM;                                            

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


