-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:     Charge pump test bench
-- 
-- Description:
-- The test-bench shows how a digital PLL is built up from
-- a phase-frequency detector, a charge pump, an integrator
-- and a VCO.
--
-- Literature:
-- 
--
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    VCO(ANALOG)
-- WORK                    CP(SIMPLE)
-- WORK                    PFD(A1)
-- -----------------------------------------------------------
--
-- Source:
-- section7.3.3_cp/tb/bench.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity BENCH is end BENCH;


architecture TEST_DPLL of BENCH is

  --constants
  constant  REF_TONE   :  REAL:= -30.0;
  constant  REF_FREQ   :  REAL:= 1.0E06;
  constant  REF_KF     :  REAL:= 5.0e10;
  constant  LO_TONE    :  REAL:= -30.0;
  constant  LO_FREQ    :  REAL:= 1.0e06;
  constant  LO_KF      :  REAL:= 5.0e10;
  constant  MIXER_GAIN :  REAL:=   0.0;
  constant  COMP_POINT :  REAL:= -20.0;
  constant  IIP3_POINT :  REAL:= -30.0;
  constant  CORNER_FREQ:  REAL:= 5.0e3;
  constant  R_S        :  REAL:= 50.0;

  terminal N_REFCLK    : ELECTRICAL;
  terminal N_CLK       : ELECTRICAL;
  terminal N_CTRL      : ELECTRICAL;
  terminal N_TP        : ELECTRICAL;
  terminal N_TPINT     : ELECTRICAL;
  terminal N_TPOUT     : ELECTRICAL;
  terminal N_VDD       : ELECTRICAL;
  terminal N_VSS       : ELECTRICAL;
  signal S_1, S_2: BIT;
  --branch quantities
  quantity V_VDD    across I_VDD    through N_VDD;
  quantity V_VSS    across I_VSS    through N_VSS;
  quantity V_VCTRL  across I_VCTRL  through N_CTRL;
  quantity V_RTPIN  across I_RTPIN  through N_TP;
  quantity V_RTPOUT across I_RTPOUT through N_TPINT to N_TPOUT;
  quantity V_TPOUT  across I_TPOUT  through N_TPINT;
  quantity VOUT1,VOUT2: REAL;

  signal CTRL: REAL:=0.0;
begin                                          
                  
  V_VDD == 5.0;
  V_VSS == -5.0;
  CTRL<=0.0, 5.0E-5 after 100US, 0.0 after 200US;
  V_VCTRL == CTRL;
  
  SRC: entity VCO(ANALOG)
    generic map ( AMPL_DBM => REF_TONE,
                 FREQ_0 => REF_FREQ,
		 K_FREQ => REF_KF)
    port map (P_CTRL=>N_CTRL,
	      P_OUT=>N_REFCLK,
	      VDD=>N_VDD,
	      GND=>ELECTRICAL_REF);
                         
  MIX: entity PFD(A1)
    generic map ( THD => 0.0)
    port map (S_UP=>S_1,
	      S_DOWN=>S_2,
	      P_REFCLK=>N_REFCLK,
	      P_CLK=>N_CLK,
	      GND=>ELECTRICAL_REF);

  CHARGEPUMP: entity CP(SIMPLE)
    generic map (R_ISRC=>100.0E4)
    port map (S_UP=>S_1,
	      S_DOWN=>S_2,
	      P_OUT=>N_TP,
	      VDD=>N_VDD,
	      VSS=>N_VSS);

  V_RTPIN == R_S * I_RTPIN;
  V_RTPOUT == R_S * I_RTPOUT;
  VOUT1'DOT==MATH_2_PI*CORNER_FREQ*V_RTPIN;
  VOUT2==0.05*V_RTPIN;
  V_TPOUT==2.0*(VOUT1+VOUT2);

  LO: entity VCO(ANALOG)
    generic map ( AMPL_DBM => LO_TONE,
                 FREQ_0 => LO_FREQ,
		 K_FREQ => LO_KF)
    port map (P_CTRL=>N_TPOUT,
	      P_OUT=>N_CLK,
	      VDD=>N_VDD,
	      GND=>ELECTRICAL_REF);
                                         
end architecture TEST_DPLL;
