-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Digital-to-analog converter
-- 
-- Description:
-- This model describes a digital to analog converter. The conversion
-- is done in a digital process at every change of the digital input
-- signal. The transition to the analog domain includes a 'RAMP attribute
-- where the transistion times can be parameterized.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- d2a.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity D2A is
  generic (PMAX_DBM : REAL := -100.0;   -- maximum power amplitude [dBm]
           TRISE    : REAL := 1.0E-9;   -- transition time for rising edge [s]
           TFALL    : REAL := 1.0E-9;   -- transition teim for falling edge [s]
	   ROUT     : REAL := 50.0      -- output resistance [Ohm], MIN: >0.0
           );
  port ( terminal A   : ELECTRICAL;     -- analog output 
	 terminal GND : ELECTRICAL;     -- analog ground
         signal   D   : in BIT_VECTOR   -- digital input
	 );
begin
  assert ROUT > 0.0
    report "rout must be > 0.0"
    severity error;
end entity D2A;

architecture BHV_RF of D2A is
    
  constant PA  : REAL := 10**((PMAX_DBM-30.0)/10.0); 
  constant VA  : REAL := SQRT(PA * 2.0 * ROUT);
  constant VSB : REAL := VA/(2.0**D'LENGTH-1.0);
  quantity VOUT across IOUT through A to GND; 
  signal   VALUE      : real := 0.0;

begin

  -- conversion process
  process (D) is
    variable NUMBER  : INTEGER;
    begin
      NUMBER := 0;
      for I in D'HIGH downto D'LOW loop
        NUMBER := 2 * NUMBER;
        if D(I) = '1' then
          NUMBER := NUMBER + 1;
        end if;
      end loop;
         
      VALUE <= REAL(NUMBER)*VSB;

  end process;

  -- output impedance
  VOUT == 2.0*VALUE'RAMP(TRISE, TFALL) + ROUT * IOUT;

end architecture BHV_RF;
