-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Additive white Gaussian noise channel
-- 
-- Description:
-- This model describes channel model which includes effects
-- of additive white Gaussian noise in the time domain.
-- Two uniform distributed random numbers are produced that
-- are formed to Gaussian distribution. After digital generation
-- the signal is converted to the analog domain and lowpass
-- filtered.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- channel.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity CHANNEL is
  generic (PS_DBM   : REAL := -100.0; -- power of input signal
	   S_TO_N   : REAL :=  100.0; -- signal to noise ratio
	   FS_NOISE : REAL := 1.0E06; -- frequency of noise signal, MIN: >0.0
	   TD       : REAL := 0.0;    -- time delay for input signal
	   RIN      : REAL := 50.0;   -- input resistance
	   ROUT     : REAL := 50.0    -- output resistance
	   );
  port (terminal P_in  : ELECTRICAL;  -- input terminal
	terminal P_out : ELECTRICAL   -- output terminal
	);
begin
  assert FS_NOISE>0.0
    report "fs_noise must be > 0.0"
    severity error;
end entity CHANNEL;

architecture AWGN of CHANNEL is

  constant PERIOD: REAL := 1.0/FS_NOISE/2.0;
  constant PN_DBM: REAL := PS_DBM - S_TO_N;
  constant PN    : REAL := 10**((PN_DBM-30.0)/10.0); 
  constant VN    : REAL := SQRT(PN * ROUT);
  terminal N_INT: ELECTRICAL;
  quantity V_RIN  across I_RIN  through P_IN;
  quantity V_INT  across I_INT  through N_INT;
  quantity V_ROUT across I_ROUT through N_INT to P_OUT;
  quantity V_NOISE, V_LPF: real;
  
  signal S_NOISE: REAL := 0.0;

begin
  --time discrete noise generation
  process is
    variable X1,X2,X: REAL := 0.0;
    variable SD1: POSITIVE := 111;
    variable SD2: POSITIVE := 333;
  begin
    UNIFORM(SD1, SD2, X1);     -- uniform gives a value 0<x<1
    UNIFORM(SD1, SD2, X2);     -- defined in ieee.math_real   
    X:=VN*COS(MATH_2_PI*X1)*SQRT(-2.0*LOG(X2));
    S_NOISE<=X;
    wait for PERIOD;
  end process;

  --d-to-a conversion
  V_NOISE == S_NOISE;
  break on S_NOISE;

  --lowpass filtering
  V_LPF == V_NOISE'LTF((0 => 1.0), (1.0, 1.0/MATH_2_PI/FS_NOISE/2.0));

  --input impedance
  V_RIN == RIN * I_RIN;

  --time delay and noise addition
  V_INT == 2.0*(V_RIN'DELAYED(TD) + V_LPF);

  --output impedance
  V_ROUT == ROUT * I_ROUT;

end architecture AWGN;


