-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Phase-locked loop
-- 
-- Description:
-- This model describes a linear analog Phase-locked loop.
-- It consists of a multiplying mixer, an oscillator and
-- a lowpass filter. The characteristics of the blocks
-- can be parameterized.
--
-- Literature:
-- Best, R.: Phase-locked loops: design, simulation, and applications.
-- 5th edition, McGraw-Hill, 2003
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    VCO(ANALOG)
-- WORK                    MIXER(RF)
-- WORK                    LOWPASS(BHV_RF)
-- -----------------------------------------------------------
--
-- Source:
-- pll.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                         
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity PLL is
  generic ( VCO_TONE_DBM : REAL := -30.0;
	    VCO_FREQ_0   : REAL := 1.0E06;
	    VCO_K_FREQ   : REAL := 1.0E09;
	    MIX_GAIN_DB  : REAL := 0.0;
	    MIX_IIP3     : REAL := -10.0;
	    LPF_GAIN_DB  : REAL := 0.0;
	    LPF_3DBFREQ  : REAL := 1.0E5;
	    LPF_GRADE    : INTEGER := 1);
  port (terminal P_REFCLK: ELECTRICAL;
	terminal P_OUT   : ELECTRICAL;
	terminal P_OUTCLK: ELECTRICAL;
	terminal P_VDD   : ELECTRICAL;
	terminal P_GND   : ELECTRICAL);
end entity PLL;

architecture LINEAR of pll is

  terminal N_TP, N_TPOUT: ELECTRICAL;
  
begin                                          
  -- phase detector                
  MIX: entity MIXER(RF)
    generic map (GP_DB   => MIX_GAIN_DB,                       
		 IP3_DBM => MIX_IIP3)
    port map (P_IN => P_REFCLK,
	      P_OUT => N_TP,
	      P_CLOCK => P_OUTCLK,
	      VDD => P_VDD,
	      GND => P_GND);

  -- loop filter
  LPF: entity LOWPASS_FILTER (BHV_RF)
    generic map ( GAIN => LPF_GAIN_DB,
		  FG   => LPF_3DBFREQ,
		  GRAD => LPF_GRADE)
    port map (N_TP, P_GND, P_OUT, P_GND);

  --  voltage controlled oscillator
  OSC: entity VCO(ANALOG)
    generic map ( AMPL_DBM => VCO_TONE_DBM,
                 FREQ_0    => VCO_FREQ_0,
		 K_FREQ    => VCO_K_FREQ)
    port map (P_CTRL => P_OUT,
	      P_OUT => P_OUTCLK,
	      VDD => P_VDD,
	      GND => P_GND);

end architecture LINEAR;

