-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      First Order Lowpass and Pole-Zero Filter
-- 
-- Description:
-- The model describes an analog filter. The architecture LP
-- realizes a first order lowpass filter with cut-off frequency
-- FC. The architecture PZ realizes a pole-zero filter with
-- pole frequency FC and zero frequency ZF.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- filter.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
use IEEE.MATH_REAL.all;

entity FILTER is
   generic (GAIN          : REAL := 1.0; -- gain
            FC            : REAL := 1.0; -- cut-off frequency [Hz]
            ZF            : REAL := 1.0  -- zero frequency [Hz]
           );
   port    (terminal INP  : ELECTRICAL;   -- input terminal
            terminal OUTP : ELECTRICAL    -- output terminal
           );
begin
   assert FC > 0.0 and ZF > 0.0           -- parameter conditions
     report "FC and ZF > 0.0 required."
     severity ERROR;
end entity FILTER;


architecture LP of FILTER is               -- lowpass description
  quantity VIN  across INP;                -- open input branch
  quantity VOUT across IOUT through OUTP;  -- output branch
begin
  VOUT == VIN'LTF((0 => GAIN), (0 => 1.0, 1 => 1.0/MATH_2_PI/FC));
end architecture LP;


architecture PZ of FILTER is
  quantity VIN  across INP;                  -- input branch
  quantity VOUT across IOUT through OUTP;    -- output branch
begin
  VOUT == GAIN*VIN'LTF(
         (0 => 1.0, 1 => 1.0/MATH_2_PI/ZF),  -- numerator
         (0 => 1.0, 1 => 1.0/MATH_2_PI/FC)   -- denominator
          );
end architecture PZ;
