-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Lossless Line
-- 
-- Description:
-- The model describes an ideal lossless line based on Branin's
-- approach.
-- 
-- Literature:
-- Branin Jr., F. H.:
-- Transient Analysis of Lossless Transmission Lines.
-- Proc. of the IEEE 55(1967), pp. 2012-2013.
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- -----------------------------------------------------------
--
-- Source:
-- line.vhd
-- -----------------------------------------------------------


library IEEE_proposed;
    use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
   
entity LINE is
    generic (Z0 : REAL := 50.0;     -- characteristic impedance [Ohm], MIN: >0.0
             TD : REAL := 1.0e-3    -- transmission delay [s], MIN: >0.0
            );
    port (terminal T1 : ELECTRICAL; -- plus terminal at port 1
          terminal T2 : ELECTRICAL; -- minus terminal at port 1
          terminal T3 : ELECTRICAL; -- plus terminal at port 2
          terminal T4 : ELECTRICAL  -- minus terminal at port 2
         );
end entity LINE;
   
architecture LOSSLESS of LINE is
    quantity V1 across I1 through T1 to T2;
    quantity V2 across I2 through T3 to T4;
    quantity ER, ES    : REAL;
begin
    V1 == Z0*I1 + ES;
    V2 == Z0*I2 + ER;
    ER == 2.0*V1'DELAYED(TD) - ES'DELAYED(TD);
    ES == 2.0*V2'DELAYED(TD) - ER'DELAYED(TD);
end architecture LOSSLESS;
