-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Bessel Lowpass Filter
-- 
-- Description:
-- The architecture BESSEL_2 realizes a second order analog Bessel
-- lowpass. The architecture BESSEL_4 realizes a fourth order analog
-- Bessel lowpass. The cut-off frequency is FC.
-- The architecture BESSEL_2_DIGITAL realizes a second order digital
-- Bessel lowpass. The cut-off frequency is FC. The sampling
-- frequency is 4*FC.
-- 
-- Literature:
-- Tietze, U,; Schenk, C.: Halbleiter-Schaltungstechnik.
-- Heidelberg: Springer-Verlag, 2002.
--
-- Schruefer, E.: Signalverarbeitung.
-- Muenchen, Wien: Carl Hanser Verlag, 1990.
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- lowpass.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
    use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
    use IEEE.MATH_REAL.all;
   
entity LOWPASS is
    generic (FC              : REAL;          -- cut-off frequency [Hz], MIN: >0.0
             GAIN            : REAL := 1.0    -- voltage gain
            );
    port (terminal INP  : ELECTRICAL;         -- controlling terminal
          terminal OUTP : ELECTRICAL          -- output terminal
            );
end entity LOWPASS;
   
architecture BESSEL_2 of LOWPASS is
    constant W   : REAL := MATH_2_PI*FC;
    constant NUM : REAL_VECTOR := (0 => 1.0);
    constant DEN : REAL_VECTOR 
             := (1.0, 1.3617/W, 0.6180/W/W); 
    quantity VIN  across              INP;
    quantity VOUT across IOUT through OUTP;
begin
    VOUT == GAIN*VIN'LTF(NUM,DEN);
end architecture BESSEL_2;

       
architecture BESSEL_4 of LOWPASS is
    quantity VIN  across              INP;
    quantity VOUT across IOUT through OUTP;
    quantity Q1   : REAL;
 
    constant W    : REAL        := MATH_2_PI*FC;
    constant NUM1 : REAL_VECTOR := (0 => GAIN);
    constant DEN1 : REAL_VECTOR
                    := (1.0, 1.3397/W, 0.4889/W/W);
    constant NUM2 : REAL_VECTOR := (0 => 1.0);
    constant DEN2 : REAL_VECTOR
                    := (1.0, 0.7743/W, 0.3890/W/W);
begin
    Q1   == VIN'LTF(NUM1, DEN1); 
    VOUT == VIN'LTF(NUM2, DEN2);
end architecture BESSEL_4;   

   
architecture BESSEL_2_DIGITAL of LOWPASS is
    constant FS      : REAL := 4.0*FC;  -- only in this case
    constant TSAMPLE : REAL := 1.0/FS;
    constant NUM : REAL_VECTOR 
             := (0.3356, 0.6712, 0.3356);
    constant DEN : REAL_VECTOR 
             := (1.0000, 0.2564, 0.0860); 
    quantity VIN  across              INP;
    quantity VOUT across IOUT through OUTP;
begin
    VOUT == GAIN*VIN'ZTF(NUM,DEN, TSAMPLE, 0.0);
end architecture BESSEL_2_DIGITAL;

