-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:     Wobble generator
-- 
-- Description:
-- The model describes a sine wave where the frequency sweeps
-- over a given range from STARTFREQ to STOPFREQ. The amplitude
-- of the sine wave is given in term sof power.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- wobble.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity WOBBLE is
  generic (
    INITDELAY: REAL:=0.0;    -- initial time delay before oscillation starts [s]
    STARTFREQ: REAL:=1.0;    -- initial frequency where sweep starts [Hz]
    SWEEPRATE: REAL:=1.0;    -- rate of change for frequency sweep [Hz/s]
    STOPFREQ : REAL:=1.0e7;  -- end frequency where sweep stops [Hz]
    AMP_DBM  : REAL:=-100.0; -- power amplitude of sine wave [dBm]
    ROUT     : REAL:= 50.0   -- output resistance [Ohm]
    );
  port (
    terminal P : ELECTRICAL; -- positive terminal
    terminal M : ELECTRICAL  -- negative terminal
    );
end entity WOBBLE;

architecture BHV of WOBBLE is
  constant AMP_LIN : REAL:= 10**((AMP_DBM-30.0)/10.0); 
  constant AMP     : REAL:= SQRT(AMP_LIN * 2.0 * ROUT);
  terminal N_INT   : ELECTRICAL;
  quantity PHI     : REAL;                           
  quantity EFFFREQ : REAL := STARTFREQ;
  quantity V_ROUT across I_ROUT through P to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to M;
begin
 
  if NOW > INITDELAY and EFFFREQ < STOPFREQ use
    EFFFREQ == STARTFREQ + SWEEPRATE*(NOW-INITDELAY);
  elsif EFFFREQ >= STOPFREQ use
    EFFFREQ == STOPFREQ;
  else
    EFFFREQ == STARTFREQ;
  end use;

  if DOMAIN = QUIESCENT_DOMAIN USE
    PHI == 0.0;
  else
    PHI'DOT == MATH_2_PI*EFFFREQ;
  end use;

-- signal source
  V_SRC == 2.0 * AMP * SIN(PHI);
 
-- output port resistance
  V_ROUT == ROUT * I_ROUT;
end architecture BHV;
