-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Switch
-- 
-- Description:
-- This model describes a switch that is built up with a changeable
-- resistance. Transistions are modeled using the 'RAMP attribute
-- where the transistion times can be parametrized.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    STD_LOGIC_1164
-- -----------------------------------------------------------
--
-- Source:
-- switch.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.STD_LOGIC_1164.all;

entity SWITCH is

  generic (RON        : REAL := 1.0E-3;  -- resistance when switch is closed
           ROFF       : REAL := 1.0E6;   -- resistance when switch is open, MIN: ROFF>RON
           TON        : REAL := 1.0E-6;  -- transition time for rising edge, MIN: >=0.0
           TOFF       : REAL := 1.0E-6   -- transition time for falling edge, MIN: >=0.0
	   );
  port (terminal P    : ELECTRICAL;      -- positive terminal
        terminal M    : ELECTRICAL;      -- negative terminal
        signal   C    : in STD_LOGIC     -- control signal
	);
begin

  assert (TON >= 0.0) 
    report "ton >= 0 required."
    severity error;
  assert (TOFF >= 0.0) 
    report "ton >= 0 required."
    severity error;
  assert ROFF > RON
    report "roff > ron required."
    severity error;

end entity SWITCH;

architecture RAMP of SWITCH is

  signal   R_VAL : REAL := ROFF;
  quantity V_SW across I_SW through P    to M;

begin

  R_VAL <= RON  when C = '1' else
           ROFF when C = '0';

  V_SW == R_VAL'RAMP(TON, TOFF)*I_SW;

end architecture RAMP;

