-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:     Voltage controlled oscillator
-- 
-- Description:
-- The test-bench describes a voltage controlled oscillator
-- which includes an analog and a digital architecture.
-- The analog realization uses a sine wave with tunable frequency
-- while the digital realization uses a process that produces
-- a square wave.
--
-- Literature:
-- 
--
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- vco.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity VCO is 
  generic (DC        : real := 0.0;     -- dc offset of oscillator [V]
           AMPL_DBM  : real := -100.0;  -- amplitude of sine wave [dBm]
           FREQ_0    : real := 1.0e03;  -- free running frequency [Hz], MIN: >0.0
	   PHASE_0   : real := 0.0;     -- constant phase offset [rad]
           K_FREQ    : real := 1.0e06;  -- sensitivity of voltage input [Hz/V]
           RIN       : real := 50.0;    -- input resistance [Ohm], MIN: >0.0
           ROUT      : real := 50.0     -- output resistance [Ohm], MIN: >0.0
	   );
  port (terminal P_CTRL : ELECTRICAL;   -- control pin
	terminal P_OUT  : ELECTRICAL;   -- output pin 
	terminal VDD    : ELECTRICAL;   -- supply voltage
	terminal GND    : ELECTRICAL    -- reference pin
	);    
begin
  assert RIN>0.0;
  assert ROUT>0.0;
  assert FREQ_0>0.0;
end entity VCO;   

architecture ANALOG of VCO is
  constant AMPL_LIN : REAL:= 10**((AMPL_DBM-30.0)/10.0); 
  constant AMPL     : REAL:= SQRT(AMPL_LIN * 2.0 * ROUT); 
  terminal N_INT    : ELECTRICAL;
  quantity V_RIN  across I_RIN  through P_CTRL to GND;
  quantity V_ROUT across I_ROUT through P_OUT to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to GND;
  quantity PHI_W : REAL;
begin
-- input impedance
  V_RIN  == RIN * I_RIN;

-- tunable oscillator
  if DOMAIN = QUIESCENT_DOMAIN use
    PHI_W == 0.0;
  else
    0.0 == PHI_W'DOT - MATH_2_PI*K_FREQ*V_RIN;
  end use;
  
  V_SRC == 2.0 * (DC + AMPL * SIN(NOW * MATH_2_PI*FREQ_0 + PHI_W + PHASE_0));

-- output impedance
  V_ROUT == ROUT * I_ROUT;

end architecture ANALOG;                                            

--+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

architecture DIGITAL of VCO is
  constant AMPL_LIN : REAL:= 10**((AMPL_DBM-30.0)/10.0); 
  constant AMPL     : REAL:= SQRT(AMPL_LIN * 2.0 * ROUT);
  signal STATE      : bit:='0';
  signal FACTOR     : real:=0.0;
  terminal N_INT    : ELECTRICAL;
  quantity V_RIN  across I_RIN  through P_CTRL to GND;
  quantity V_ROUT across I_ROUT through P_OUT to N_INT;
  quantity V_SRC  across I_SRC  through N_INT to GND;
begin

  process (STATE,V_RIN'ABOVE(-FREQ_0/K_FREQ))
    VARIABLE F: REAL;
  begin
    if STATE='1' then 
      FACTOR<=1.0;
    else
      FACTOR<=-1.0;
    end if;
    F := FREQ_0 + K_FREQ*V_RIN;
    if F>0.0 then
      STATE <= not STATE after 0.5/F*SEC;
    end if;    
  end process;
  break on STATE;
  
-- input impedance
  V_RIN  == RIN * I_RIN;

-- tunable oscillator
  V_SRC == 2.0*(DC+FACTOR*AMPL);

-- output impedance
  V_ROUT == ROUT * I_ROUT;

end architecture digital;                                            


