-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Power measurement unit
-- 
-- Description:
-- This model describes a power measurement unit with two different
-- architectures. The architecture feedthrough has no load resistance
-- but measures the current flowing though it. The architecture
-- terminating includes a termination resistor to match the
-- resistance of the preceding block. The power is measured over
-- an integration period of T_INTEG which is a model parameter
-- and can therefore be adjusted.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- power.vhd
-- -----------------------------------------------------------

library IEEE,IEEE_proposed;
  use IEEE.MATH_REAL.all;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;

entity POWER_METER IS   
  generic (T_INTEG: real:= 1.0e-03;   -- integration period [s]
	   R_TERM : real:= 50.0       -- termination resistance [Ohm]
	   );      
  port (terminal P_1: ELECTRICAL;     -- first measurement pin
	terminal P_2: ELECTRICAL;     -- second measurement pin
	quantity P_DBM: out REAL      -- output power
	);
end POWER_METER;

architecture FEEDTHROUGH of POWER_METER is

  quantity V_12 across I_12 through P_1 to P_2;
  quantity V_1  across              P_1;
  quantity P_INT      : REAL;
  signal INT_OLD, INT : REAL:=0.0; 

begin

  V_12 == 0.0;

  process is
    begin
      INT_OLD<=P_INT;
      INT<=(P_INT-INT_OLD)/T_INTEG;
      wait for T_INTEG;
  end process;
  break on INT;

  if INT>0.0 use
    P_DBM == 10.0*LOG10(INT) + 30.0;
  else
    P_DBM == -300.0;
  end use;

  if DOMAIN=QUIESCENT_DOMAIN use
    P_INT == 0.0;
  else
    P_INT'DOT == V_1 * I_12;
  end use;
  
end architecture FEEDTHROUGH;


architecture TERMINATING of POWER_METER is

  quantity V_12 across I_12 through P_1 to P_2;
  quantity P_INT: REAL;
  signal INT_OLD, INT: REAL:=0.0; 
  
begin

  V_12 == I_12 * R_TERM;
                  
  process is
    begin
      INT_OLD<=P_INT;
      INT<=(P_INT-INT_OLD)/T_INTEG;
      wait for T_INTEG;
  end process;
  break on INT;

  if INT>0.0 use
    P_DBM == 10.0*LOG10(INT)+30.0;
  else
    P_DBM == -300.0;               
  end use;

  if DOMAIN=QUIESCENT_DOMAIN use
    P_INT == 0.0;
  else
    P_INT'DOT == V_12 * I_12;
  end use;
  
end architecture TERMINATING;
