-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:     PLL test bench
-- 
-- Description:
-- The test-bench shows the frequency synchronization of the
-- internal PLL signal with the external reference.
--
-- Literature:
-- 
--
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    PLL(LINEAR)
-- WORK                    VCO(ANALOG)
-- WORK                    LOWPASS(BHV_RF)
-- -----------------------------------------------------------
--
-- Source:
-- section7.5_pll/tb/bench.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity BENCH is end BENCH;

architecture TEST_PLL of BENCH is

  constant  REF_TONE   :  REAL:= -30.0;
  constant  REF_FREQ   :  REAL:= 1.0e06;
  constant  REF_KF     :  REAL:= 1.0e09;
  constant  PHASE_INI  :  REAL:= -math_pi/2.0;
  constant  LO_TONE    :  REAL:= -30.0;
  constant  LO_FREQ    :  REAL:= 1.0e06;
  constant  LO_KF      :  REAL:= 1.0e09;
  constant  MIXER_GAIN :  REAL:=   12.0;
  constant  MIXER_IIP3 :  REAL:= -10.0;
  constant  FILTER_GAIN:  REAL:=   0.0;
  constant  FILTER_GRAD:  INTEGER:=  1;
  constant  CORNER_FREQ:  REAL:= 1.0e5;
  constant  R_S        :  REAL:= 50.0;

  terminal N_REFCLK: ELECTRICAL;
  terminal N_CLK: ELECTRICAL;
  terminal N_CTRL: ELECTRICAL;
  terminal N_VDD: ELECTRICAL;
  terminal N_TP: ELECTRICAL;
  terminal N_TPOUT: ELECTRICAL;
  quantity V_VDD   across I_VDD   through N_VDD;
  quantity V_VCTRL across I_VCTRL through N_CTRL;
  quantity V_OUT   across I_OUT   through N_TPOUT;

  signal CTRL: REAL:=0.0;
  
begin                                          
  V_VDD == 2.7;
  CTRL<=0.0, 5.0E-5 after 100US, 1.0E-4 after 200US;
  V_VCTRL == CTRL;
  V_OUT == R_S * I_OUT;
  
  SRC: entity VCO(ANALOG)
    generic map ( AMPL_DBM => REF_TONE,
                 FREQ_0    => REF_FREQ,
 		 PHASE_0   => PHASE_INI,
 		 K_FREQ    => REF_KF)
    port map (P_CTRL => N_CTRL,
	      P_OUT => N_REFCLK,
	      VDD => N_VDD,
	      GND => ELECTRICAL_REF);
    
  FREQ_SYNTH: entity PLL(LINEAR)
    generic map (VCO_TONE_DBM => LO_TONE,
		 VCO_FREQ_0   => LO_FREQ,
	         VCO_K_FREQ   => LO_KF,
	         MIX_GAIN_DB  => MIXER_GAIN,
	         MIX_IIP3     => MIXER_IIP3,
	         LPF_GAIN_DB  => FILTER_GAIN,
	         LPF_3DBFREQ  => CORNER_FREQ,
	         LPF_GRADE    => FILTER_GRAD)
    port map (P_REFCLK => N_REFCLK,
	      P_OUT => N_TP,
	      P_OUTCLK => N_CLK,
	      P_VDD => N_VDD,
	      P_GND => ELECTRICAL_REF);

  LPF2: entity LOWPASS_FILTER (BHV_RF)
    generic map ( GAIN => FILTER_GAIN,
		  FG   => CORNER_FREQ,
		  GRAD => 2)
    port map (INP => N_TP,
	      INN => ELECTRICAL_REF,
	      OUTP => N_TPOUT,
	      OUTN => ELECTRICAL_REF);

end architecture TEST_PLL;
