-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Operational amplifier
-- 
-- Description:
-- This model describes an macromodel of an operational amplifier.
-- Individual blocks for the input and output stages, the frequency
-- response, and the gain stage are included.
--
-- Literature:
-- Boyle, G.R.; Cohn, B.M.; Pederson, D.O.; Solomon, J.E.:
-- Macromodeling of integrated circuit operational amplifiers.
-- IEEE J. Solid-State Circuits SC-9 (1974) 6, 353 - 363
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- opamp.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity OPAMP is 
  generic (AVD0 : REAL := 106.0;  -- DC differential gain [dB]
           FP1  : REAL := 5000.0; -- dominant pole [Hz]
           FP2  : REAL := 2.0E6;  -- pole frequency [Hz]
           ROUT : REAL := 75.0    -- output resistance [Ohm]
           );
  port (terminal INP: ELECTRICAL; -- input plus terminal
        terminal INM: ELECTRICAL; -- input minus terminal
        terminal OUTP: ELECTRICAL -- output terminal
        );
  end entity OPAMP;

  architecture MACRO of OPAMP is

  -- Input stage
    quantity V_IN across I_IN through INP to INM;
  -- Frequency Response
    constant NUM_2 : REAL_VECTOR := (0 => 1.0);
    constant DEN_2 : REAL_VECTOR := (1.0, 1.0/MATH_2_PI/FP2);
    quantity q_fr3 : REAL;

  -- Gain stage
    constant AVD0_VAL : REAL := 10.0**(Avd0/20.0);
    constant NUM_1 : REAL_VECTOR := (0 => 1.0);
    constant DEN_1 : REAL_VECTOR := (1.0, 1.0/MATH_2_PI/FP1);
    quantity Q_SUM : REAL;
    quantity Q_FP1 : REAL;

  -- Output stage
    quantity v_out across i_out through outp;

  begin

  -- Input stage
    I_IN == 0.0;

  -- Frequency Response
    Q_FR3 == V_IN'LTF(NUM_2, DEN_2);

  -- Gain stage
    Q_SUM == AVD0_VAL*Q_FR3;
    Q_FP1 == Q_SUM'LTF(NUM_1, DEN_1);

  -- Output stage
    I_OUT == (V_OUT - Q_FP1)/ROUT;

end architecture MACRO;






