-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Mixer
-- 
-- Description:
-- This model describes an analog mixer and includes the multiplier
-- stage and amplifier properties like gain, compression and
-- transfer function. Resistances at the input and output
-- terminals are modeled as well.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- -----------------------------------------------------------
--
-- Source:
-- mixer.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;

entity MIXER is 
  generic (GP_DB     : REAL := 0.0;       -- power gain of stage [dB]
	   IP3_DBM   : REAL := -20.0;     -- input referenced IP3  [dBm]
           FNOISE_DB : REAL := 0.0;       -- noise figure [dB]
	   RIN       : REAL := 50.0;	  -- input resistance [Ohm], MIN: >0.0
           RIN_CLK   : REAL := 50.0;	  -- input resistance at clock input [Ohm], MIN: >0.0
           ROUT      : REAL := 50.0;	  -- output resistance [Ohm], MIN: >0.0
           FG        : REAL := REAL'HIGH  -- dominant pole [Hz], MIN: >0.0
	   );
  port (terminal P_IN   : ELECTRICAL;  -- input pin
	terminal P_OUT  : ELECTRICAL;  -- output pin
	terminal P_CLOCK: ELECTRICAL;  -- local oscillator pin
	terminal VDD    : ELECTRICAL;  -- supply voltage
	terminal GND    : ELECTRICAL   -- reference terminal
	);      
begin
  assert RIN>0.0;
  assert RIN_CLK>0.0;
  assert ROUT>0.0;
  assert FG>0.0;
end entity MIXER;

architecture RF of MIXER is
  constant GP_LIN   : REAL:= 10**(GP_DB/10.0);           -- linear value of power gain
  constant IP3_LIN  : REAL:= 10**((IP3_DBM-30.0)/10.0);   -- linear value of ip3
  constant A        : REAL:= SQRT(GP_LIN*ROUT/RIN);    -- linear value of voltage gain
  constant IP3      : REAL:= SQRT(IP3_LIN*2.0*RIN);     -- linear value of ip3 voltage
  constant B        : REAL:= A/(IP3*IP3)*4.0/3.0;        -- third order coefficent
  constant INMAX    : REAL:= SQRT(A/(3.0*B));            -- maximum input voltage for clipping
  constant OUTMAX   : REAL:= 2.0*A/3.0*INMAX;            -- output voltage at clipping
  constant TENV     : REAL:= 300.0;
  constant NOISE_LVL: REAL:= 4.0*1.38e-23*(10**(FNOISE_DB/10.0)-1.0)*TENV*RIN;   
  terminal IN_G     : ELECTRICAL;
  terminal OUT_G    : ELECTRICAL;
  terminal OUT_F    : ELECTRICAL;
  terminal U_OUT    : ELECTRICAL;
  quantity V_NOISE across I_NOISE through P_IN to IN_G;
  quantity V_RIN   across I_RIN   through IN_G to GND;
  quantity V_OUTG  across I_OUTG  through OUT_G to GND;
  quantity V_CLK   across I_CLK   through P_CLOCK to GND;
  quantity V_OUTF  across I_OUTF  through OUT_F to GND;
  quantity V_LIM   across I_LIM   through U_OUT to GND;      
  quantity V_ROUT  across I_ROUT  through U_OUT to P_OUT;              
                                                                   
begin

  -- noise figure
  V_NOISE == 0.0;

  -- input impedances
  V_RIN == RIN * I_RIN;
  V_CLK == RIN_CLK * I_CLK;
  
  -- gilbert cell
  V_OUTG == V_RIN * V_CLK;

  -- lna
  V_OUTF == V_OUTG'LTF((0 => 1.0), (1.0, 1.0/MATH_2_PI/FG));

  if abs(V_OUTF)<INMAX use
    V_LIM == 2.0*(A - B*V_OUTF*V_OUTF)*V_OUTF;
  elsif V_OUTF > 0.0 use
    V_LIM == 2.0*OUTMAX;
  else
    V_LIM == -2.0*OUTMAX;
  end use;

  -- output impedance
  V_ROUT == ROUT * I_ROUT; 

end architecture RF;


                          

                                                 
