-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Phase-locked loop
-- 
-- Description:
-- This model describes a linear analog Phase-locked loop.
-- It consists of a multiplying mixer, an oscillator and
-- a lowpass filter. Additional splitter blocks are inserted
-- compared to the PLL realization in section 7.5.
-- The characteristics of the blocks can be parameterized.
--
-- Literature:
-- Best, R.: Phase-locked loops: design, simulation, and applications.
-- 5th edition, McGraw-Hill, 2003
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    VCO(ANALOG)
-- WORK                    MIXER(RF)
-- WORK                    LOWPASS(BHV_RF)
-- -----------------------------------------------------------
--
-- Source:
-- pll.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity PLL is
  generic ( VCO_TONE_DBM : REAL := -30.0;
	    VCO_FREQ_0   : REAL := 1.0E06;
	    VCO_PHASE_0  : REAL := 0.0;
	    VCO_K_FREQ   : REAL := 1.0E09;
	    MIX_GAIN_DB  : REAL := 0.0;
	    MIX_IIP3     : REAL := -10.0;
	    LPF_GAIN_DB  : REAL := 0.0;
	    LPF_3DBFREQ  : REAL := 1.0E5;
	    LPF_GRADE    : INTEGER := 1);
  port (terminal P_REFCLK, P_OUT, P_OUTCLK, P_VDD, P_GND: ELECTRICAL);
end entity PLL;

architecture LINEAR of PLL is

  terminal N_TP, N_TPOUT, N_VCO, N_VCOOUT, N_CLK: ELECTRICAL;
  
begin                                          
  -- phase detector                
  MIX: entity MIXER(RF)
    generic map (GP_DB   => MIX_GAIN_DB,                       
		 IP3_DBM => MIX_IIP3)
    port map (P_IN => P_REFCLK,
	      P_OUT => N_TP,
	      P_CLOCK => N_CLK,
	      VDD => P_VDD,
	      GND => P_GND);

  -- loop filter
  LPF: entity LOWPASS_FILTER (BHV_RF)
    generic map ( GAIN => LPF_GAIN_DB,
		  FG   => LPF_3DBFREQ,
		  GRAD => LPF_GRADE)
    port map (N_TP, P_GND, N_TPOUT, P_GND);

  -- splitter1                
  SP1: entity SPLITTER(IDEAL)
    port map (P_I=>N_TPOUT, P_O1=>P_OUT, P_O2=>N_VCO, GND=>P_GND);

  --  voltage controlled oscillator
  OSC: entity VCO(ANALOG)
    generic map ( AMPL_DBM => VCO_TONE_DBM,
                 FREQ_0    => VCO_FREQ_0,
		 PHASE_0   => VCO_PHASE_0,
		 K_FREQ    => VCO_K_FREQ)
    port map (P_CTRL => N_VCO,
	      P_OUT => N_VCOOUT,
	      VDD => P_VDD,
	      GND => P_GND);

  -- splitter2                
  SP2: entity SPLITTER(IDEAL)
    port map (P_I=>N_VCOOUT, P_O1=>P_OUTCLK, P_O2=>N_CLK, GND=>P_GND);

end architecture LINEAR;

