-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      WLAN receiver
-- 
-- Description:
-- This model describes the complete RF front end of a WLAN
-- receiver according to IEEE802.11a standard. The receiver
-- has a double conversion architecture where the same
-- local oscillator frequency is applied twice in order to
-- mix the signal from passband down to baseband.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    VCO(ANALOG)
-- WORK                    LNA(RF)
-- WORK                    MIXER(RF)
-- WORK                    LOWPASS(BHV_RF)
-- WORK                    PLL(LINEAR)
-- WORK                    SPLITTER(IDEAL)
-- WORK                    SHIFTER(IDEAL)
-- -----------------------------------------------------------
--
-- Source:
-- receiver.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;                                            
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity RECEIVER IS
  port (terminal P_RF, P_I, P_Q, P_CTRL, P_VDD, P_GND: ELECTRICAL);
end entity RECEIVER;

architecture DOUBLE_CONV of RECEIVER is

  constant  LO_TONE    :  REAL:=  -30.0;
  constant  LO_FREQ    :  REAL:=  2.6E09;
  constant  LO_PHASE   :  REAL:=  MATH_PI_OVER_2;
  constant  LO_KF      :  REAL:=  1.0E09;
  constant  LNA_GAIN   :  REAL:=  6.0;
  constant  LNA_IIP3   :  REAL:=  -5.0;
  constant  MIXER_GAIN :  REAL:=  5.0 + 46.0;
  constant  MIXER_IIP3 :  REAL:=  5.0;
  constant  FILTER_GAIN:  REAL:=  0.0;
  constant  FILTER_GRAD:  INTEGER:=  5;
  constant  CORNER_LPF :  REAL:= 9.0E6;
  constant  BBAMP_GAIN :  REAL:= 18.0;
  constant  BBAMP_IIP3 :  REAL:= 20.0;
  constant  R_S        :  REAL:=  50.0;
  constant  CORNER_HPF1:  REAL:=  1.0e9;
  constant  C_hpf1     :  REAL:=  1.0/(MATH_2_PI*CORNER_HPF1*R_S);

  terminal N_REFCLK, N_CLK, N_CLKIF, N_CLKRF, N_CLKI, N_CLKQ, N_TP: ELECTRICAL;
  terminal N_RFAMP, N_RFHPF, N_IF, N_IFI, N_IFQ: ELECTRICAL;
  terminal N_IFHPFI, N_BBI, N_BBHPFI, N_BBFILTI: ELECTRICAL;
  terminal N_IFHPFQ, N_BBQ, N_BBHPFQ, N_BBFILTQ: ELECTRICAL;

  quantity V_C1    across I_C1    through N_RFAMP to N_RFHPF;  
  quantity V_C2I   across I_C2I   through N_IFI to N_IFHPFI;  
  quantity V_C2Q   across I_C2Q   through N_IFQ to N_IFHPFQ;  

begin

---------------- LO generation ----------------------
  -- local oscillator
  LO: entity VCO(ANALOG)
    generic map (AMPL_DBM => LO_TONE,
                 FREQ_0   => LO_FREQ,
		 K_FREQ   => LO_KF)
    port map (P_CTRL=>P_CTRL, P_OUT=>N_REFCLK, VDD=>P_VDD, GND=>P_GND);

  -- frequency synthesizer PLL
  FREQ_SYNTH: entity PLL(LINEAR)
    generic map (VCO_TONE_DBM => LO_TONE,
		 VCO_FREQ_0   => LO_FREQ,
		 VCO_PHASE_0  => LO_PHASE,
	         VCO_K_FREQ   => LO_KF)
    port map (P_REFCLK=>N_REFCLK, P_OUT=>N_TP, P_OUTCLK=>N_CLK,
	      P_VDD=>P_VDD, P_GND=>P_GND);

  -- clock splitter
  SPC: entity SPLITTER(IDEAL)
    port map (P_I=>N_CLK, P_O1=>N_CLKRF, P_O2=>N_CLKIF, GND=>P_GND);

  -- phase shifter
  PHS: entity SHIFTER(IDEAL)
        generic map (FREQ => LO_FREQ)
        port map    (INP => N_CLKIF, OUT0 => N_CLKI, OUT90 => N_CLKQ);

---------------- receiver rf section ----------------------
  -- low noise amplifier rf 
  LNA1: entity LNA(RF)
    generic map ( GP_DB   => LNA_GAIN,                           
		  IP3_DBM => LNA_IIP3)
    port map (P_IN=>P_RF, P_OUT=>N_RFAMP, VDD=>P_VDD, GND=>P_GND);

  -- high pass rf 
  I_C1 == C_HPF1 * V_C1'DOT;

  -- mixer rf 
  MIX1: entity MIXER(RF)
    generic map ( GP_DB   => MIXER_GAIN,                           
		  IP3_DBM => MIXER_IIP3)
    port map (P_IN=>N_RFHPF, P_OUT=>N_IF, P_CLOCK=>N_CLKRF, VDD=>P_VDD, GND=>P_GND);
                         
  -- signal splitter
  SPS: entity SPLITTER(IDEAL)
    port map (P_I=>N_IF, P_O1=>N_IFI, P_O2=>N_IFQ, GND=>P_GND);

---------------- receiver baseband section i path ----------------------
  -- high pass filter if i path 
  I_C2I == C_HPF1 * V_C2I'DOT;

  -- mixer i path 
  MIX2I: entity MIXER(RF)
    generic map ( GP_DB   => MIXER_GAIN,                           
		  IP3_DBM => MIXER_IIP3)
    port map (P_IN=>N_IFHPFI, P_OUT=>N_BBI, P_CLOCK=>N_CLKI, VDD=>P_VDD, GND=>P_GND);

  -- low pass filter baseband i path 
  BBFiltI: entity LOWPASS_FILTER (BHV_RF)
             generic map (GAIN => FILTER_GAIN,
			  FG   => CORNER_LPF,
			  GRAD => FILTER_GRAD)
             port map (N_BBI, P_GND, N_BBFILTI, P_GND);

  -- amplifier baseband i path 
  BBAmpI: entity LNA(RF)
    generic map ( GP_DB   => BBAMP_GAIN,
		  IP3_DBM => BBAMP_IIP3)
    port map (P_IN=>N_BBFILTI, P_OUT=>P_I, VDD=>P_VDD, GND=>P_GND);
                         
---------------- receiver baseband section q path ----------------------
  -- high pass filter if q path 
  I_C2Q == C_HPF1 * V_C2Q'DOT;

  -- mixer q path 
  MIX2Q: entity MIXER(RF)
    generic map ( GP_DB   => MIXER_GAIN,                           
		  IP3_DBM => MIXER_IIP3)
    port map (P_IN=>N_IFHPFQ, P_OUT=>N_BBQ, P_CLOCK=>N_CLKQ, VDD=>P_VDD, GND=>P_GND);
                         
  -- low pass filter baseband q path 
  BBFiltQ: entity LOWPASS_FILTER (BHV_RF)
             generic map (GAIN => FILTER_GAIN,
			  FG   => CORNER_LPF,
			  GRAD => FILTER_GRAD)
             port map (N_BBQ, P_GND, N_BBFILTQ, P_GND);

  -- amplifier baseband q path 
  BBAmpQ: entity LNA(RF)
    generic map ( GP_DB   => BBAMP_GAIN,                           
		  IP3_DBM => BBAMP_IIP3)
    port map (P_IN=>N_BBFILTQ, P_OUT=>P_Q, VDD=>P_VDD, GND=>P_GND);
                         
end architecture DOUBLE_CONV;

