-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      Transmitter
-- 
-- Description:
-- This model describes the RF front end of a transmitter to be
-- used with the WLAN receiver model. Only one signal path is
-- implemented.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    VCO(ANALOG)
-- WORK                    MIXER(RF)
-- WORK                    LOWPASS(BHV_RF)
-- -----------------------------------------------------------
--
-- Source:
-- transmit.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity TRANSMIT is
  port (TERMINAL P_RF, P_I, P_Q, P_CTRL,
	         P_VDD, P_GND: ELECTRICAL);
end entity TRANSMIT;

architecture SIMPLE of TRANSMIT is

  --constants
  constant  CAR_TONE   :  REAL:= -30.0;
  constant  CAR_FREQ   :  REAL:= 5.2E09;
  constant  CAR_PHASE  :  REAL:= MATH_PI_OVER_2;
  constant  LO_KF      :  REAL:=  1.0E09;
  constant  MIXER_GAIN :  REAL:=  0.0 + 46.0;
  constant  MIXER_IIP3 :  REAL:=  5.0;
  constant  R_S        :  REAL:=  50.0;
  constant  IN_FREQMAX :  REAL:= 8.0e06;
  constant  C_LPF      :  REAL:= 1.0/(MATH_2_PI*IN_FREQMAX*R_S);

  terminal N_CARRIER, N_XMTI: ELECTRICAL;

begin                                          

----------------  LO generation ----------------------
  
  -- local oscillator
  LO: entity VCO(ANALOG)
    generic map ( AMPL_DBM => CAR_TONE,
		  FREQ_0   => CAR_FREQ,
		  PHASE_0  => CAR_PHASE,
		  K_FREQ   => LO_KF)
    port map (P_CTRL=>P_CTRL, P_OUT=>N_CARRIER,
	      VDD=>P_VDD, GND=>P_GND);

  -- transmit filter
  LPF: entity LOWPASS_FILTER (BHV_RF)
    generic map ( GAIN => 0.0,
		  FG   => 8.0E6,
		  GRAD => 1)
    port map (P_I, P_GND, N_XMTI, P_GND);
  
  -- mixer rf 
  MIX1: entity MIXER(RF)
    generic map ( GP_DB   => MIXER_GAIN,                           
		  IP3_DBM => MIXER_IIP3)
    port map (P_IN=>N_XMTI, P_OUT=>P_RF, P_CLOCK=>N_CARRIER,
	      VDD=>P_VDD, GND=>P_GND);
                         
end architecture SIMPLE;
