-- -----------------------------------------------------------
-- 
-- Additional material to the book
-- Modeling and Simulation for RF System Design
-- 
--
-- THIS MODEL IS LICENSED TO YOU "AS IT IS" AND WITH NO WARRANTIES, 
-- EXPRESSED OR IMPLIED. THE AUTHORS SPECIFICALLY DISCLAIM ALL IMPLIED 
-- WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
-- THEY MUST NOT HAVE ANY RESPONSIBILITY FOR ANY DAMAGES, FINANCIAL OR
-- LEGAL CLAIMS WHATEVER.
-- -----------------------------------------------------------

-- Name:      WLAN receiver test bench
-- 
-- Description:
-- This model describes a complete transmission system including
-- a pseudorandom binary source, digital-to-analog conversion,
-- a transmitter which modulates the signal on a carrier frequency,
-- and a receiver which demodulates the signal from the carrier.
--
-- Literature:
-- 
-- Dependencies: 
-- -----------------------------------------------------------
-- Logical Library         Design unit
-- -----------------------------------------------------------
-- IEEE_proposed           ELECTRICAL_SYSTEMS
-- IEEE                    MATH_REAL
-- WORK                    PRBS(SHIFT_REGISTER)
-- WORK                    D2A(BHV_RF)
-- WORK                    TRANSMIT(IDEAL)
-- WORK                    RECEIVER(DOUBLE_CONV)
-- -----------------------------------------------------------
--
-- Source:
-- section9.5_wlan/tb/bench.vhd
-- -----------------------------------------------------------

library IEEE, IEEE_proposed;
  use IEEE_proposed.ELECTRICAL_SYSTEMS.all;
  use IEEE.MATH_REAL.all;
  use WORK.all;

entity BENCH is end BENCH;

architecture TEST_REC of BENCH is

  constant  CAR_TONE   :  REAL:= -30.0;
  constant  CAR_FREQ   :  REAL:= 5.2E09;
  constant  CAR_PHASE  :  REAL:= -MATH_PI_OVER_2;
  constant  LO_KF      :  REAL:=  1.0E09;
  constant  MIXER_GAIN :  REAL:=  0.0 + 46.0;
  constant  MIXER_IIP3 :  REAL:=  5.0;
  constant  IN_TONE    :  REAL:= -30.0;
  constant  IN_FREQ    :  REAL:= 5.0E06;
  constant  IN_FREQMAX :  REAL:= 8.0E06;
  constant  R_S        :  REAL:=  50.0;

  terminal N_VDD, N_ANEG, N_CTRL, N_RF: ELECTRICAL;
  terminal N_I, N_I_IN, N_Q, N_Q_IN: ELECTRICAL;

  quantity V_VDD   across I_VDD   through N_VDD;
  quantity V_VANEG across I_VANEG through N_ANEG;
  quantity V_VCTRL across I_VCTRL through N_CTRL;
  quantity V_R2I   across I_R2I   through N_I;  
  quantity V_R2Q   across I_R2Q   through N_Q;

  signal CTRL: REAL:=0.0;
  signal S_BIT: BIT_VECTOR(0 to 0);

begin                                          
                  
  V_VDD == 2.7;
  CTRL<=0.0;
  V_VCTRL == CTRL;
  V_VANEG == -0.01;
  
  SRC3: entity PRBS(SHIFT_REGISTER)
    generic map (POLYGRAD=>3,
		 BIT_TIME=>1.0/IN_FREQ*1SEC)
    port map (BIT_OUT=>S_BIT(0));

  CONV: entity D2A(BHV_RF)
    generic map(PMAX_DBM => IN_TONE)
    port map (D=>S_BIT, A=>N_I_IN, GND=> N_ANEG);
  
  XMT: entity TRANSMIT(SIMPLE)
    port map (P_RF   => N_RF,
	      P_I    => N_I_IN,
	      P_Q    => N_Q_IN,
	      P_CTRL => N_CTRL,
	      P_VDD  => N_VDD,
	      P_GND  => ELECTRICAL_REF);

  REC: entity RECEIVER(DOUBLE_CONV)
    port map (P_RF   => N_RF,
	      P_I    => N_I,
	      P_Q    => N_Q,
	      P_CTRL => N_CTRL,
	      P_VDD  => N_VDD,
	      P_GND  => ELECTRICAL_REF);
  
  V_R2I == R_S * I_R2I;
  V_R2Q == R_S * I_R2Q;

end architecture TEST_REC;

