%------------------------------------------
%  Demonstration of finite memory controls
%------------------------------------------
clc;
clear all;

%------------------------
% System parameters
%------------------------
A = [ 0.9305   0   0.1107        ; 
        0.0077 0.9802  -0.0173  ; 
        0.0142 0  0.8953 ]         ;
B = [0.0217 0.2510    ;
       0.0192 -0.0051   ;
       0.0247 0.0030 ]  ;

G = [1 1 1]' ;
C = [ 1 0 0;0 1 0 ] ; 
Q_w = 0.02^2 ;
R_v = 0.04^2*eye(2) ;
Q = eye(3)  ;
R = eye(2)  ;
Q_f = 1000*eye(3)  ;

%------------------------
% Control and filter horizons
%------------------------
F_horizon =10 ;
C_horizon =10 ;

%------------------------
% Simulation parameters
%------------------------
N_sample = 800 ;
N_order =  size(A,1) ;
N_horizon = 10 ;
intial_state = [1 1 -1 ]' ;

x_fmc = intial_state ;
x_lq = intial_state ;

%% Initial values for Kalman filter
KF_x_hat = [0 0 0]' ;
P = 0.5*eye(N_order) ;

%------------------------
% Variables for saving simulation results
%------------------------

real_state = zeros(N_order , N_sample) ;
real_state(:,1) = intial_state ;

LQG_state = zeros(N_order, N_sample) ;
LQG_state(:,1)=KF_x_hat ;

FMC_state = zeros(N_order, N_sample) ;

measurements = zeros ( 2*N_sample) ;          
inputs = zeros ( 2*N_sample) ;               

%------------------------
% Parameters for Uncertainties
%------------------------
delta_A = 0.05*[1 0 0 ; 0 1 0 ; 0 0 0.1 ];
delta_C = 0.05*[0.1  0  0 ; 0  0.1   0 ];
dist_start = 300  ;
dist_stop = 501 ;

%------------------------
% Compute FMC and LQC 
%------------------------
[H  L ] = fmc ( A , B , C , G , Q , R , Q_f , Q_w , R_v , F_horizon , C_horizon ) ;
N = zeros( size(A,1) , size(B,2) ) ; 
[K,S,E] = dlqr(A,B,eye(3),eye(2),N) ;


%------------------------
% Simulation starts here !
%------------------------
for i = 1 : N_sample-1
	sys_noise=randn(1)*0.01;
   	mea_noise=randn(2,1)*0.01;
   
   	% FMC
   	x_temp_fmc = x_fmc ;
   	if i > F_horizon 
      	u = H * (measurements(2*i-(2*F_horizon-1) : 2*i))'+ L * (inputs(2*i-(2*F_horizon-1):2*i))' ;
   	else % outputs and inputs are not enough
      	u = H(:,1:2*i) * (measurements(1 : 2*i))'+ L(:,1:2*i) * (inputs(1:2*i))' ;   
   	end

   	if( i > dist_start & i < dist_stop )
   		x_fmc = (A+delta_A)*x_fmc + B*u + G*sys_noise ; % next state
   		y_1  =   (C+delta_C)*x_temp_fmc + mea_noise ; 
	else 
      	x_fmc = A*x_fmc + B*u +G*sys_noise ; %next state
   		y_1 = C*x_temp_fmc + mea_noise ;        
      end
   	FMC_state(:,i+1) = x_fmc ;
   	measurements(2*i-1:2*i) = y_1 ;   
   	inputs(2*i-1:2*i)  = u ;
   
   	% LQG 
   	u = -K*KF_x_hat ;
   	x_temp_lqg = x_lq ;
   	if( i > dist_start & i < dist_stop )
         	x_lq = (A+delta_A)*x_lq + B * u + G*sys_noise ;    		
         	y_2 = (C+delta_C)*x_temp_lqg + mea_noise ; 
	else 
      	x_lq  = A * x_lq + B * u + G * sys_noise ;	
   		y_2  = C * x_temp_lqg + mea_noise ;        
      end
      LQG_state(:,i+1) = x_lq;
      
	% Kalman filter for LQG
   	KF_x_hat = A * KF_x_hat + A* P *C' * inv( R_v + C * P * C')*( y_2 - C*KF_x_hat) + B *u ;
   	P = A * inv( eye(N_order) + P * C' * inv(R_v) * C) * P * A' + G * Q_w * G';
end

%------------------------
% Simulation results are shown here.
%------------------------

time = 1:N_sample ;

plot( time , LQG_state(1,1:N_sample),'r' ,time ,FMC_state(1,1:N_sample),'b' );
title('First state trajectory');
xlabel('Time');
legend('LQG','FMC');
figure;
plot( time , LQG_state(2,1:N_sample),'r' ,time ,FMC_state(2,1:N_sample),'b' );
title('Second state trajectory');
xlabel('Time');
legend('LQG','FMC');
figure;
plot( time , LQG_state(3,1:N_sample),'r' ,time ,FMC_state(3,1:N_sample),'b' );
title('Third state trajectory');
xlabel('Time');
legend('LQG','FMC');

