%------------------------------------------
%
% 		Demonstration : 
%
% 		1. Steady-state H infinity filter
% 		2. Steady-state Kalman Filter 
% 		3. RHKF 
% 		4. H inf FIR filter 
% 		5. mixed H2/H infty FIR filter 
%
%------------------------------------------
clc;
clear all;

%----------------------
%  2nd-order System
%----------------------
A=[0.9950  0.0998;
  -0.0998  0.9950];
B=[0.1;0.1];
C=[1 0];
G_det=[1 0;1 0]; 
D =[0 1];
Q= 1;
R =1;

%% Set to 1 for the simulation of H infty filter
Hinf_FIR_Flag = 1; 
mixed_H_Flag = 1; 

G_sto = [1;1];   %% G for Stochastic formulaion
G_det=[1 0;1 0]; %% G for Deterministic formulation
D =[0 1];
Q = 0.05; %% System noise covariance
R = 0.05; %% Measurement noise covariance
Q_aug = daug(Q,R);  %% Augmented noise covariance

[n,n]=size(A);
[m,n]=size(C);
[n,p]=size(B);
[n,q]=size(G_det);

%% We can see that matrices 'G' and 'D' satisfy the following relations
%% D*G' = 0 (zero matrix), D*D' = I (identity matrix)

if(Hinf_FIR_Flag == 1) 
   load H_FIR; 
end 

if(mixed_H_Flag == 1) 
   load H_mixed; 
end 

Nmax = 10;   %% Maximal horizon length
hXih = [];

invA = inv(A);
if(n==2)
  C_bar = C*inv(A);
  B_bar = -C*inv(A)*B;
  G_bar_sto = -C*inv(A)*G_sto;
  G_bar_det = -C*inv(A)*G_det;
  D_bar = D;
  Q_bar = Q;
  R_bar = R;
  Q_aug_bar = Q_aug;
end

for N = n:1:Nmax,
    [dummy, ncB_bar] = size(B_bar);
    [dummy, ncG_bar_det] = size(G_bar_det);
    [dummy, ncG_bar_sto] = size(G_bar_sto);
    B_bar = [B_bar   -C_bar*invA*B;
             zeros(m,ncB_bar)  -C*invA*B];
    G_bar_sto = [G_bar_sto   -C_bar*invA*G_sto;
             zeros(m,ncG_bar_sto)  -C*invA*G_sto];         
    G_bar_det = [G_bar_det   -C_bar*invA*G_det;
             zeros(m,ncG_bar_det)  -C*invA*G_det];         
    C_bar =[C_bar;C]*invA;
    D_bar = daug(D_bar, D);
    
    Q_bar = daug(Q_bar, Q);
    R_bar = daug(R_bar, R);
    Q_aug_bar = daug(Q_aug_bar, Q_aug);
    
    Xi_gHFIR = G_bar_det*G_bar_det' + D_bar*D_bar';
    Xi_rhkf = G_bar_sto*Q_bar*G_bar_sto' + R_bar;
    H_rhkf = inv(C_bar'*inv(Xi_rhkf)*C_bar)*C_bar'*inv(Xi_rhkf);
 end
 
[nHr nHc] = size(H_rhkf);
error_var = trace(H_rhkf*Xi_rhkf*H_rhkf'); 


%------------------------
% Obtain the steady-state Kalman filter
%------------------------
[K_kal,P_kal,Z,E] = DLQE(A,[1;1],C,Q,R);  		% K_kal, Kalman filter gain
K_kal =  A*P_kal*C'*inv(R+C*P_kal*C');     		% Steady-state Kalman filter 

%------------------------
% Now, let's compute the filter gain and the minimum 
% H infinity norm of  the conventional H infinity filter using LMI 
%------------------------
setlmis([]);
X=lmivar(1,[n 1]);
Y=lmivar(2,[n m]);
r=lmivar(1,[1 1]);   %  H infinity norm bound

lmiterm([-1 1 1 X],1,1);      % LMI #1: X > 0

% Diagonal Terms
lmiterm([2 1 1 X],-1,1);    %  (1,1) = -X
lmiterm([2 2 2 r],-1,1);    %  (2,2) = -r*I
lmiterm([2 3 3 X],-1,1);    %  (3,3) = -X
lmiterm([2 4 4 r],-1,1);    %  (3,3) = -r*I

% Nondiagonal Terms 
lmiterm([2 3 1 X],1,A);     
lmiterm([2 3 1 Y],-1,C);    %  (3,1) = X*A-Y*C
lmiterm([2 3 2 X],1,G_det);     %  
lmiterm([2 3 2 Y],-1,D);    %  (3,2) = X*G_det - Y*D
lmiterm([2 4 1 0],1);       %  (4,1) = I

%------------------------
% Solve LMI
%------------------------
hinf_lmi=getlmis;
n_lmi = decnbr(hinf_lmi);
c = zeros(n_lmi,1);
for j=1:n_lmi,
   rj = defcx(hinf_lmi,j,r);
   c(j) = rj;
end

[jopt, xfeas]= mincx(hinf_lmi, c', [1e-6 300 0 0 0]);

X = dec2mat(hinf_lmi,xfeas,X);
Y = dec2mat(hinf_lmi,xfeas,Y);
r = dec2mat(hinf_lmi,xfeas,r);  %% Obtaind minimum H infinity norm
K_Hinf=inv(X)*Y;                     %% H infity filter gain 

Y= zeros(Nmax,1);
y=Y(end);
x = [0;0];
x_rhkf = [0;0]; 		%% guarnateed H FIR
X_rhkf = [];
x_kal = [0;0];
X_kal = [];
x_mixed = [0;0];
X_mixed = [];
x_HFIR = [0;0]; 	
X_HFIR = [];
x_Hinf=[0;0];
X_Hinf =[];
X=[];
W=[];
del = 0.1; 	%% If you want to add some uncertainty to the system, change this value.
randn('seed',8);

%------------------------
% Simulation starts here !
%------------------------
for i=0:1:300,
   %% Disturbance Generation
   if((i>=0) & (i<=300))
      %w =sqrt(1)*[randn;randn];  		%% Random noise
      %w=0.1*[sin(i/20);cos(i/20)]; 		%% Sinusoidal disturbance
      w=0.1*[exp(-i/30);exp(-i/30)]; 	%% Exponentially decreasing disturbance
      %w=[0.1;0.1]; 						%% Constant disturbance
   else
      w=[0;0];
   end
   W=[W w];
   
   %% Uncertainty Generation
   if((i>=50) & (i<=100))
	   del_A=[0 0;0 del];
      del_C=[0   0];
   else
      del_A=zeros(2,2);
      del_C=[0 0];
   end
   
   X=[X x]; %% True state data saving
   x_Hinf = A*x_Hinf + K_Hinf*(y-C*x_Hinf);  %% Conventioanl H infinity filter
   x_kal = A*x_kal + K_kal*(y-C*x_kal); %% Steady-state Kalman filter 
   x_rhkf = H_rhkf*Y;  %% Guaranteed H infinity FIR filter
   
   if(mixed_H_Flag)
      x_mixed = H_mixed*Y;
   end
   
   if(Hinf_FIR_Flag)
      x_HFIR = H_FIR*Y;  %% Hinf FIR filter using SDP 
   end   
   
   y = (C+del_C)*x + D*w;  %% Output equation
   Y=[Y(2:Nmax);y]; %% Measurement data update
   x = (A+del_A)*x + G_det*w;

   
   X_Hinf=[X_Hinf x_Hinf];
   X_rhkf=[X_rhkf x_rhkf];
   X_kal = [X_kal x_kal];
   
   if(mixed_H_Flag)
      X_mixed = [X_mixed x_mixed];
   end
   
   if(Hinf_FIR_Flag)
      X_HFIR=[X_HFIR x_HFIR];
   end   
   
end

I=0:1:300;
%------------------------
% Simulation results are shown here.
%------------------------
if(~Hinf_FIR_Flag & mixed_H_Flag)
  figure; 
  plot(I,X(1,:)-X_Hinf(1,:),'c',I, X(1,:)-X_kal(1,:),'m', I,X(1,:)-X_rhkf(1,:),'k', I, X(1,:)-X_HFIR(1,:),'b',I,X(1,:)-X_mixed(1,:),'r');
  legend('H_\infty filter', 'Kalman filter', 'RHKF', 'H_\infty FIR', 'mixed H filter');
  title('Errors in x_1');
  xlabel('Time');
  figure; 
  plot(I,X(2,:)-X_Hinf(2,:),'c',I, X(2,:)-X_kal(2,:),'m', I,X(2,:)-X_rhkf(2,:),'k', I, X(2,:)-X_HFIR(2,:),'b',I,X(2,:)-X_mixed(2,:),'r');
  legend('H_\infty filter', 'Kalman filter', 'RHKF', 'H_\infty FIR','mixed H filter');
  title('Errors in x_2');
  xlabel('Time');
end

if(Hinf_FIR_Flag & mixed_H_Flag)
  figure; 
  plot(I,X(1,:)-X_Hinf(1,:),'--',I, X(1,:)-X_kal(1,:),'-.', I,X(1,:)-X_mixed(1,:),'-');
  legend('H_\infty filter', 'H_2 filter', 'mixed H_2/H_\infty filter');
  title('Estimation error of x_1');
  xlabel('Time');
  grid; 
  figure; 
  plot(I,X(2,:)-X_Hinf(2,:),'--',I, X(2,:)-X_kal(2,:),'-.',I,X(2,:)-X_mixed(2,:),'-');
  legend('H_\infty filter', 'H_2 filter','mixed H_2/H_\infty filter');
  title('Estimation error in x_2');
    xlabel('Time');
  grid;
end

if(~Hinf_FIR_Flag)
  plot(I,X(2,:)-X_rhkf(2,:),'r',I,X(2,:)-X_Hinf(2,:),'b');
end

if(Hinf_FIR_Flag)
   figure;
  plot(I,X(2,:),'r',I, X_rhkf(2,:),'k:',I,X_Hinf(2,:),'b', I, X_HFIR(2,:),'k');
  legend('real state','RHKF', 'Conventional H_\infty', 'H_\infty FIR');
  title('State trajectory');
  xlabel('Time');
end