! File: chap5.f90
! Date: June 1, 2002
! Update: March, 2003
!         May,   2004
!         September, 2004
! Author: O. Kardaun
!
!
! Purpose: 
! making illustrations for Chap. 5 of Classical Methods of Statistics
! (Springer-Verlag, 2004)

! (C) 2004 O. Kardaun, free software, released under GNU general public license, 
!     Version 2, June 1991, see http://www.gnu.org/copyleft/gpl.html  

!     Utilises routines from public domain KOMPLOT library (Version 8.0),
!     authored by J. Kraak, and available on internet, see
!     http://www.rug.nl/rc/hpcv/visualisation/komplot

! Note: the postscript files that are produced are named discrim1*.*ps

! set precision parameters
!===============================================================================
!------------------------------------------------------------------------------+
Module set_precision

   Integer, Parameter :: wp = Kind(1.0e0) !single
  ! Integer, Parameter :: wp = Kind(1.0d0) !double
   Integer, Parameter :: sp = Kind(1.0e0) !single
 
End Module set_precision
!------------------------------------------------------------------------------+
!===============================================================================


! set debug flag
!===============================================================================
!------------------------------------------------------------------------------+
Module debug_decl
  ! variable declaration debug flag
   Logical            :: debug
  ! debug: indicator whether debug mode is set on (= .TRUE.) or off (= .FALSE.)
End Module debug_decl
!------------------------------------------------------------------------------+
!===============================================================================


! Module declaring a number of variables for making komplot graphs
!===============================================================================
!------------------------------------------------------------------------------+
Module kmplt_graph_decl
  Use set_precision                  
 
   Integer            :: solid_dot, empty_dot, star
   Integer            :: light_star, dot, solid_diamond
   Integer            :: num_dens, num_fmt
   Real(sp)           :: xs,xl,xh,ys,yl,yh,sf_ma,sf_an,sf_an_sec
   Real(sp)           :: xp,yp,height,angle,glab_xpct,glab_ypct 

  ! meaning of variables:
  ! xs: size of x-axis in cm
  ! xl: lower  value of x-axis (data -units)
  ! xh: higher value of x-axis
  ! ys: size of y-axis in cm
  ! yl: lower  value of y-axis (data -units)
  ! yh: higher value of y-axis
  ! sf_ma: scale factor to blow up marker   symbols
  ! sf_an: scale factor to blow up annotate symbols
  ! sf_an_sec: scale factor to blow up annotate symbols for secondary axis
  ! xp: x-axis position of a string to be plotted
  ! yp: y-axis position of a string to be plotted
  ! height: height of a string to be plotted
  ! angle: angle of a string to be plotted
  ! glab_xpct: shift (in percentage of frame) in x-direction, used for xp
  ! glab_ypct: shift (in percentage of frame) in y-direction, used for yp
End Module  kmplt_graph_decl
!------------------------------------------------------------------------------+
!===============================================================================

! declare four variables used in discriminant overview plots
!===============================================================================
!------------------------------------------------------------------------------+
Module discrim_var4_decl
   Use set_precision                   
   Integer, Parameter  :: nmax=1500
   Real(wp) :: elip(nmax),elnel(nmax),elpinj(nmax),lseplima(nmax) 
End Module discrim_var4_decl
!------------------------------------------------------------------------------+
!===============================================================================



! Module containing subroutine(s) for making multiple plots on A4 format
!===============================================================================
!------------------------------------------------------------------------------+
Module pageplots
  ! Starting date: March, 2003
  ! Main author: O. Kardaun

  ! Contains subroutines: discrim_var4
  ! Contains (generic) function: hlog

  ! Imported variables and procedures:
   Use set_PRECISION                   ! from komplot_ext_mod.f90 
   Use komplot_ext, Only: sec_axis     ! from komplot_ext_mod.f90 

   Implicit None

  ! Generic function name, for real and integer input:
   Interface hlog
   Module Procedure rhlog, ihlog
   End Interface hlog

Contains

!===============================================================================
Pure Function ihlog(iar) ! RESULT(ihlog)
  ! caculates logartihm of absolute value, 
  ! while returning huge for negative values the the argument

   Use set_PRECISION               

   Implicit None

   Integer, Intent(IN) :: iar
   Real(wp) :: ihlog
!-End of header (function ihlog)------------------------------------------------
  !
  If ( Abs(iar)>0 ) Then
     ihlog=Log(Abs(Real(iar)))
     Else
     ihlog=Huge(1.0_wp)
  Endif
End Function ihlog
!===============================================================================

!===============================================================================
Pure Function rhlog(ar) ! RESULT(rhlog)
  ! calculates logarithm of absolute value, 
  ! while returning huge for negative values the the argument

   Use set_PRECISION               
 
   Implicit None

   Real(wp), Intent(IN) :: ar
   Real(wp) :: rhlog
!-End of header (function rhlog)------------------------------------------------
  !
   If ( Abs(ar)>0 ) Then
      rhlog=Log(Abs(ar))
      Else
      rhlog=Huge(1.0_wp)
   Endif
End Function rhlog
!===============================================================================

!===============================================================================
Subroutine discrim_var4 (n, isyma,                                             &
     &                   elip, elnel, elpinj, lseplima,                        &
     &                   l_elip, h_elip, l_elnel, h_elnel,                     &
     &                   l_elpinj, h_elpinj, l_lseplima, h_lseplima,           &
     &                   n_elip, n_elnel, n_elpinj, n_lseplima,                &
     &                   nd_le_to_ve, nd_ri_to_ve, nd_le_bo_ve, nd_ri_bo_ve,   & 
     &                   nd_le_to_ho, nd_ri_to_ho, nd_le_bo_ho, nd_ri_bo_ho)   

  ! OJK: sa is for variable marker size. However, presently (March 2003)
  ! OJK: it cannot be combined with isyma, only with isym, unfortunately.

  ! Calls: <KOMPLOT: scale_dimensions, page_dimensions, suppress_border, 
  !                  scale_markers, scale_annotations, x_margins, 
  !                  shift_frame_in_page,
  !                  tick_distances, frame, gstrng
  !        <Module KOMPLOT_EXT: sec_axis
  !        <INTERNAL:  nr_tick_intervals
  ! Called by: Program DISCRIM1

  ! Imported Procedures:
   Use komplot_ext, Only: sec_axis

  ! Imported variables:
  !                           ! parameters with intent=in:
   Use kmplt_graph_decl       ! variables for making komplot graphs
   Use debug_decl             ! debug flag

  Implicit None

  ! Imported arguments
   Integer, Intent (IN)             :: n
   Integer, Intent (IN)             :: isyma(1:n)
   Real(wp), Intent (IN)            :: elip(1:n), elnel(1:n)                    
   Real(wp), Intent (IN)            :: elpinj(1:n), lseplima(1:n) 
   Real(wp), Intent (IN), Optional  :: l_elnel, h_elnel, l_elip, h_elip
   Real(wp), Intent (IN), Optional  :: l_elpinj, h_elpinj 
   Real(wp), Intent (IN), Optional  :: l_lseplima, h_lseplima
   Integer, Intent (IN), Optional   :: n_elip, n_elnel, n_elpinj, n_lseplima   
   Integer, Intent (IN), Optional   :: nd_le_to_ve, nd_ri_to_ve 
   Integer, Intent (IN), Optional   :: nd_le_bo_ve, nd_ri_bo_ve  
   Integer, Intent (IN), Optional   :: nd_le_to_ho, nd_ri_to_ho 
   Integer, Intent (IN), Optional   :: nd_le_bo_ho, nd_ri_bo_ho

  ! Meaning of parts of variable names:
  ! l_: lowest axis data value 
  ! h_: highest axis data value 
  ! n_: number of ticks
  ! nd_ : num_dens (factor to weed out ticks on secondary axis)

  ! Imported 

  ! local parameters
   Integer, Parameter :: out=6 

  ! local variables
   Integer            :: i,j,ix, iy
   Real(wp)           :: tidi_x, tidi_y
  ! meaning: tidi: tick distance 

  !-End of subroutine header ---------------------------------------------------

   debug = .True.
   If (debug) Then
      Write (out,*) '-Upon entering subroutine discrim_var4: ';
      Write (out,*) ' n, isyma '
      Write (out,*) ' elip(1:3), elnel(1:3)'
      Write (out,*) ' elpinj(1:3), lseplima(1:3):'
      Write (out,*) n, isyma(1:3)
      Write (out,*) elip(1:3),' ',elnel(1:3)
      Write (out,*) elpinj(1:3),' ',lseplima(1:3)
      Write (out,*) ' l_elnel, h_elnel, l_elip, h_elip,' 
      Write (out,*) 'l_elpinj, h_elpinj, l_lseplima, h_lseplima:'
      Write (out,*)   l_elnel, h_elnel, l_elip, h_elip
      Write (out,*)  l_elpinj, h_elpinj, l_lseplima, h_lseplima
      Write (out,*) ' End of parameter list from subroutine discrim_var4 ';
   Endif

  ! composing plots, while using the KOMPLOT package:

   Call scale_dimensions(1.0)
   Call page_dimensions(19.,24.)        ! A4 format
  !  CALL frame_style('scientific')
   Call suppress_border(1) 

  !  CALL gridlines(1,1)

   sf_an_sec =  1.15_wp ! was 1.2_wp
   glab_xpct = 15.0_wp
   glab_ypct =  8.5_wp

   Call scale_markers(sf_ma)  
   Call scale_annotations(sf_an_sec)
   Call x_margins(2.0,1.0) 

  ! size of graphs
   xs=5.75  ; ys=9.0

  ! general parameters of alphanumeric label for plot:

   height = 0.28_wp*sf_an_sec
   angle  = 0.0_wp 

  !-----------------------------------------------------------------------------
  ! plot 1 (top), basic regression variables 
  !-----------------------------------------------------------------------------
   Call x_margins(2.0,1.5)              ! left and right x-margins
   Call shift_frame_in_page(0.0,12.0)   ! left top position
   xl= l_elnel  ; xh=h_elnel  ! ne
   yl= l_elip   ; yh=h_elip   ! Ip
   tidi_x = xs/Real(n_elnel) ; tidi_y = ys/Real(n_elip)
   Call tick_distances(tidi_x,tidi_y)

   Call frame(xs,xl,xh,ys,yl,yh,'ln(ne)','ln(Ip)',' ')
   Call var_markers(' ',n,elnel,elip,isyma)

   xp=xl-glab_xpct*sf_an_sec*(xh-xl)/100.0
   yp=yh+glab_ypct*sf_an_sec*(yh-yl)/100.0
   Call gstrng (xp,yp,height,angle,'(a)') 

   Call sec_axis('h', xl, xh, yl, yh, 'ne [10^19/m^3]',                        &
  &  num_dens=nd_le_to_ho,  num_fmt=201,                                       &
  &  sf_an=sf_an_sec,debug=.False.)
   Call sec_axis('v', xl, xh, yl, yh, 'Ip [MA]',                               &
  &  num_dens=nd_le_to_ve,  num_fmt=201, lab_per_pct=17.0,                     &
  &  sf_an=sf_an_sec,debug=.False.)

  !-----------------------------------------------------------------------------
  ! plot 2 (bottom), basic regression variables 
  !-----------------------------------------------------------------------------
   Call x_margins(2.0,1.5)  
   Call shift_frame_in_page(0.0,0.0)    ! left bottom position
   xl= l_elnel  ; xh=h_elnel   ! ne
   yl= l_elpinj ; yh=h_elpinj  ! Pinj
   tidi_x = xs/Real(n_elnel) ; tidi_y = ys/Real(n_elpinj)
   Call tick_distances(tidi_x,tidi_y)
   Call frame(xs,xl,xh,ys,yl,yh,'ln(ne)','ln(Pinj)',' ')
   Call var_markers(' ',n,elnel,elpinj,isyma)

   xp=xl-glab_xpct*sf_an_sec*(xh-xl)/100
   yp=yh+glab_ypct*sf_an_sec*(yh-yl)/100.0
   Call gstrng (xp,yp,height,angle,'(b)')

   Call sec_axis('h', xl, xh, yl, yh, 'ne [10^19/m^3]',                        &
  & num_dens=nd_le_bo_ho, num_fmt=201, sf_an=sf_an_sec,                        &
  & debug=.False.)
   Call sec_axis('v', xl, xh, yl, yh, 'Pinj [MW]',                             &
  & num_dens=nd_le_bo_ve, num_fmt=201, lab_per_pct=17.0,                       &
  & sf_an=sf_an_sec,debug=.False.)

  !-----------------------------------------------------------------------------
  ! plot 3 (top), basic regression variables
  !----------------------------------------------------------------------------- 
   Call x_margins(2.0,1.5)              ! left and right x-margins
   Call shift_frame_in_page(9.5,12.0)   ! right top position

   xl=l_elpinj ; xh=h_elpinj      ! Pinj
   yl=l_elip   ; yh=h_elip        ! Ip
   tidi_x = xs/Real(n_elpinj) ; tidi_y = ys/Real(n_elip)
   Call tick_distances(tidi_x,tidi_y)
   Call frame(xs,xl,xh,ys,yl,yh,'ln(Pinj)','ln(Ip)',' ')
   Call var_markers(' ',n,elpinj,elip,isyma)

   xp=xl-glab_xpct*sf_an_sec*(xh-xl)/100.0
   yp=yh+glab_ypct*sf_an_sec*(yh-yl)/100.0
   Call gstrng (xp,yp,height,angle,'(c)') 

   Call sec_axis('h', xl, xh, yl, yh, 'Pinj [MW]',                             & 
  &  num_dens=nd_ri_to_ho, num_fmt=201,                                        &
  &  sf_an=sf_an_sec,debug=.False.)
   Call sec_axis('v', xl, xh, yl, yh, 'Ip [MA]',                               &
  &  num_dens=nd_ri_to_ve, num_fmt=201, lab_per_pct=17.0,                      &
  &  sf_an=sf_an_sec, debug=.False.)

  !-----------------------------------------------------------------------------
  ! plot 4 (bottom), basic regression variables 
  !-----------------------------------------------------------------------------
   Call x_margins(2.0,1.5) 
   Call shift_frame_in_page(9.5,0.0)    ! right bottom position

   xl= l_lseplima ; xh=h_lseplima   ! Seplima
   yl= l_elpinj   ; yh=h_elpinj     ! Pinj
   tidi_x = xs/Real(n_lseplima) ; tidi_y = ys/Real(n_elpinj)
   Call tick_distances(tidi_x,tidi_y)
   Call frame(xs,xl,xh,ys,yl,yh,'ln(seplim/a)','ln(Pinj)',' ')   
   Call var_markers(' ',n,lseplima,elpinj,isyma)

   xp=xl-glab_xpct*sf_an_sec*(xh-xl)/100
   yp=yh+glab_ypct*sf_an_sec*(yh-yl)/100.0
   Call gstrng (xp,yp,height,angle,'(d)')

   Call sec_axis('h', xl, xh, yl, yh, 'seplim/a',                              &
  &              num_dens=nd_ri_bo_ho, num_fmt=202,                            &
  &              sf_an=sf_an_sec,debug=.False.)
   Call sec_axis('v', xl, xh, yl, yh, 'Pinj [MW]',                             &
  &               num_dens=nd_ri_bo_ve, num_fmt=201, lab_per_pct=17.0,         &
  &               sf_an=sf_an_sec, debug=.False.)

   If (debug) Then
     Write (out,*) '=Upon exiting subroutine discrim_var4: ';
     Write (out,*) ' n, isyma '
     Write (out,*) ' elip(1:3), elnel(1:3)'
     Write (out,*) ' elpinj(1:3), lseplima(1:3):'
     Write (out,*) n, isyma(1:3)
     Write (out,*) elip(1:3),' ',elnel(1:3)
     Write (out,*) elpinj(1:3),' ',lseplima(1:3)
     Write (out,*) ' l_elnel, h_elnel, l_elip, h_elip,' 
     Write (out,*) 'l_elpinj, h_elpinj, l_lseplima, h_lseplima:'
     Write (out,*)   l_elnel, h_elnel, l_elip, h_elip
     Write (out,*)  l_elpinj, h_elpinj, l_lseplima, h_lseplima
    Write (out,*) ' End of parameter list from subroutine discrim_var4 ';
   Endif

End Subroutine discrim_var4
!===============================================================================

End Module pageplots
!------------------------------------------------------------------------------+
!===============================================================================

! include komplot extension module (f90 file):

   Include 'komplot_ext_mod.f90'

! Main F-90/95 program
!===============================================================================
!===============================================================================
Program chap5
  !-----------------------------------------------------------------------------
  !
  ! Main program for plots of Chap. 5 of Classical Methods of Statistics
  ! 
  ! Calls: <KOMPLOT: various komplot routines
  !        <Module pageplots: discrim_var4, FUNCTION hlog
  ! Called by: none

  ! Imported procedures:
   Use komplot_ext, Only: sec_axis             ! from komplot_ext_mod.f90         
   Use pageplots,   Only: discrim_var4, hlog

  ! Imported parameters:
   Use set_precision  

  ! Imported variables:
   Use debug_decl 
   Use discrim_var4_decl                     ! nmax and 4 real arrays for plots
   Use kmplt_graph_decl, Only: solid_dot, empty_dot, star, light_star, dot,    &
 &                             solid_diamond ! symbol constants

   Implicit None

  ! Scalar parameters:
   Integer, Parameter  :: eof=-1
   Integer, Parameter  :: out=6, df=9
   Real(wp), Parameter :: one_wp=1.0_wp
  ! n is upper bound of number of cases, p is the exact number of variables:
   Integer, Parameter :: p=18, nactual=999
   Integer, Parameter :: n_desc_lines=2

   Integer            :: ios,i,j,n, n_asd, n_jet, n_jft

 ! Character variables:
 ! Correct path name of input file has to be inserted:
   Character (len=48), Parameter :: inp='../discrim1.dat';
   Character (len=132) :: description

  ! Array variables (for data):

   Real(wp) :: x(1:nmax,1:p)
   Integer  :: tok_id(nmax), phase_id(nmax)
   Real(wp) :: shot(nmax), time(nmax), timesl(nmax), DN(nmax)
   Real(wp) :: ip(nmax),bt(nmax),nel(nmax),nev(nmax),nelohm(nmax),mgas(nmax)
   Real(wp) :: nevohm(nmax), pinj(nmax), seplim(nmax)
   Real(wp) :: rgeo(nmax), amin(nmax), kappa(nmax)
   Character (len=8) :: tok(nmax)
   Real(wp) :: elbt(nmax), elnev(nmax), elnelohm(nmax)
   Real(wp) :: elnevohm(nmax), lseplim(nmax)
   Real(wp) :: lrgeo(nmax), lamin(nmax), lkappa(nmax)
   Real(wp) :: elip_asd(nmax), elip_jft(nmax), elip_jet(nmax)
   Real(wp) :: elnel_asd(nmax), elnel_jft(nmax), elnel_jet(nmax)
   Real(wp) :: elpinj_asd(nmax), elpinj_jft(nmax), elpinj_jet(nmax)
   Real(wp) :: lseplima_asd(nmax), lseplima_jft(nmax), lseplima_jet(nmax)
   Integer  :: phase_id_asd(nmax), phase_id_jet(nmax), phase_id_jft(nmax)
   Integer  :: mgas_asd(nmax), mgas_jet(nmax), mgas_jft(nmax)
  
  ! Array variables (for KOMPLOT):
   Integer  :: isyma(nmax)

!-End of header (program discrim1)----------------------------------------------

   debug=.True.

  ! Reading ascii data file:
   If(debug) Then
      Write(out,*) 'program discrim1, starting to READ data file'
      Write(out,*) 'input file = ',inp
   End If

   Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
   If (ios /= 0) Then
      Write(out,*) 'Error in opening file : ', inp
      Write(out,*) 'ios= ', ios
   End If

   Do i=1, n_desc_lines
      Read (df,*,IOSTAT=ios) description
      Write(out,*) 'description= ', description
   End Do
   If (Debug) Then
      Write(out,*) 'description lines read'
      Write(out,*) 'i,n_desc_lines,ios = ', i,n_desc_lines,ios
   End If
   i=0
   Do While (i < nmax .And. ios == 0)
  !     write(out,*) 'In while-loop: i= ',i
      i=i+1
      Read(df,100,IOSTAT=ios) x(i,1:p)
   End Do

  ! OJK
  ! OJK 1. Somewhat of a nuisance is the somewhat cumbersome
  ! OJK    handling of missing values in FORTRAN 90/95
  ! OJK 2. Missing values are set at -9.999E19
  ! OJK 3. Notice: As required by KOMPLOT, the program presently works 
  ! OJK    only in single precision
  ! OJK 4. Do not forget that the largest real number, also for ascii input 
  ! OJK    is smaller than 10^38, otherwise you get a floating point exception
  ! OJK

100 Format(f7.3,f11.3,f7.3,1x,e10.3,f5.1,6(f7.3),2(1x,e10.3),                  & 
  &        f7.3,1x,e10.3,3(f7.3))

   If (I >= nmax .Or. ios /= eof) Then
     Write(out,*) 'Error in reading file :', inp
     Write(out,*) 'ios,eof,i,nmax= ', ios,eof,i,nmax
   End If
   n=i-1

  ! ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

  ! variable names:
   tok_id(1:n)    = Nint(x(1:n,1))
   shot(1:n)      = x(1:n,2)
   time(1:n)      = x(1:n,3)
   timesl(1:n)    = x(1:n,4)
   phase_id(1:n)  = Nint(x(1:n,5)) 
   DN (1:n)       = x(1:n,6)
   mgas (1:n)     = x(1:n,7)
   ip (1:n)       = x(1:n,8)
   bt  (1:n)      = x(1:n,9)
   nel (1:n)      = x(1:n,10)
   nev (1:n)      = x(1:n,11)
   nelohm (1:n)   = x(1:n,12)
   nevohm (1:n)   = x(1:n,13)
   pinj   (1:n)   = x(1:n,14)
   seplim (1:n)   = x(1:n,15)
   rgeo   (1:n)   = x(1:n,16)
   amin   (1:n)   = x(1:n,17)
   kappa  (1:n)   = x(1:n,18)

  ! logarithms: 
  ! (logarithms of negative values are set to huge)

   Forall (i=1:n)
     elip(i)     = hlog(ip(i))
     elbt(i)     = hlog(bt(i))
     elnel(i)    = hlog(nel(i))
     elnelohm(i) = hlog(nelohm(i))
     elnevohm(i) = hlog(nevohm(i))
     elpinj(i)   = hlog(pinj(i))
     lseplim(i)  = hlog(seplim(i))
     lrgeo(i)    = hlog(rgeo(i))
     lamin(i)    = hlog(amin(i))
     lkappa(i)   = hlog(kappa(i))
     lseplima(i) = hlog(seplim(i)/amin(i))
   End Forall

  ! special (packed) arrays per tokamak:
    elip_asd     = Pack(elip,mask=tok_id.Eq.1) 
    elip_jet     = Pack(elip,mask=tok_id.Eq.2) 
    elip_jft     = Pack(elip,mask=tok_id.Eq.3) 
    elnel_asd    = Pack(elnel,mask=tok_id.Eq.1) 
    elnel_jet    = Pack(elnel,mask=tok_id.Eq.2) 
    elnel_jft    = Pack(elnel,mask=tok_id.Eq.3)
    elpinj_asd   = Pack(elpinj,mask=tok_id.Eq.1) 
    elpinj_jet   = Pack(elpinj,mask=tok_id.Eq.2) 
    elpinj_jft   = Pack(elpinj,mask=tok_id.Eq.3)  
    lseplima_asd = Pack(lseplima,mask=tok_id.Eq.1) 
    lseplima_jet = Pack(lseplima,mask=tok_id.Eq.2) 
    lseplima_jft = Pack(lseplima,mask=tok_id.Eq.3)  
    phase_id_asd = Pack(phase_id,mask=tok_id.Eq.1) 
    phase_id_jet = Pack(phase_id,mask=tok_id.Eq.2) 
    phase_id_jft = Pack(phase_id,mask=tok_id.Eq.3)
    mgas_asd     = Pack(mgas,mask=tok_id.Eq.1) 
    mgas_jet     = Pack(mgas,mask=tok_id.Eq.2) 
    mgas_jft     = Pack(mgas,mask=tok_id.Eq.3) 
    n_asd        = Count(tok_id.Eq.1)
    n_jet        = Count(tok_id.Eq.2)
    n_jft        = Count(tok_id.Eq.3)

  Write(out,*) 'program discrim1, starting KOMPLOT'

  ! to show some of the variables read in (and some transformed):
  If (debug) Then
     Write(out,*) 'n=',n
     Write(out,*) 'x(1:3,2)', x(1:3,2)
     Write(out,*) 'elip(1:3)', elip(1:3)
     Write(out,*) 'elnel(1:3)', elnel(1:3)
     Write(out,*) 'elpinj(1:3)', elpinj(1:3)
     Write(out,*) 'lseplima(1:3)', lseplima(1:3)
     Write(out,*) 'n_asd= ', n_asd
     Write(out,*) 'i, elnel_asd, phase_id_asd, lseplima_asd:'
     Do i=1,3
        Write (out,*) i, elnel_asd(i), phase_id_asd(i), lseplima_asd(i)
     End Do
     Write(out,*) 'n_jet= ', n_jet
     Write(out,*) 'i, elip_jet, elpinj_jet, phase_id_jet, lseplima_jet:'
     Do i=1,3
        Write (out,*) i, elip_jet(i), elpinj_jet(i), phase_id_jet(i),          &
    &                 lseplima_jet(i)
     End Do
  End If

  ! set integer marker symbol array

  solid_dot     = 17
  empty_dot     = 18
  light_star    = 11
  star          = 12
  dot           = 16
  solid_diamond = 25

  ! set variable markers, all (three) tokamaks:

  Do i=1,n
      If (tok_id(i)==1) Then                 ! ASDEX
        isyma(i) = dot 
     Else If (tok_id(i)==2) Then            ! JET
        isyma(i) = solid_diamond
     Else                                   ! JFT-2M
        isyma(i) = empty_dot
     End If
  End Do

  Call discrim_var4(n,isyma,                                              &
       &            elip=elip         , elnel=elnel,                      &
       &            elpinj=elpinj     , lseplima=lseplima,                &
       &            l_elnel=0.0_wp    , h_elnel=2.5_wp,     n_elnel=5,    &
       &            l_elip=-2.5_wp    , h_elip=2.0_wp,      n_elip=9,     &
       &            l_elpinj=-2.5_wp  , h_elpinj=3.5_wp,    n_elpinj=6,   & 
       &            l_lseplima=-5.0_wp, h_lseplima=0.0_wp,  n_lseplima=5, &
       &            nd_le_to_ho=1     , nd_le_to_ve=1,                    &
       &            nd_le_bo_ho=1     , nd_le_bo_ve=1,                    &
       &            nd_ri_to_ho=2     , nd_ri_to_ve=1,                    &
       &            nd_ri_bo_ho=2     , nd_ri_bo_ve=1)

! Call komplot(50,'discrim1.eps')
  Call komplot(40,'discrim1.ps')

 ! set variable markers, ASDEX tokamak:

  Do i=1, n_asd
     If (phase_id_asd(i)==0) Then                 ! ELM-free
        isyma(i) = dot
     Else If (phase_id_asd(i)==1) Then            ! Giant ELM
        isyma(i) = solid_diamond
     Else                                         ! Small ELM
        isyma(i) = empty_dot
     End If
  End Do

  Call discrim_var4(n_asd,isyma,                                             & 
       &            elip=elip_asd     , elnel=elnel_asd,                     &
       &            elpinj=elpinj_asd , lseplima=lseplima_asd,               &
       &            l_elnel=0.0_wp      , h_elnel=2.5_wp,      n_elnel=5,    &
       &            l_elip=-1.75_wp     , h_elip=-0.75_wp,     n_elip=4,     &
       &            l_elpinj=0.25_wp    , h_elpinj=1.5_wp,     n_elpinj=5,   &
       &            l_lseplima=-1.25_wp , h_lseplima=-0.25_wp, n_lseplima=4, &
       &            nd_le_to_ho=1     , nd_le_to_ve=1,                       &
       &            nd_le_bo_ho=1     , nd_le_bo_ve=1,                       &
       &            nd_ri_to_ho=1     , nd_ri_to_ve=1,                       &
       &            nd_ri_bo_ho=1     , nd_ri_bo_ve=1)

! Call komplot(50,'discrim1_asd.eps')
  Call komplot(40,'discrim1_asd.ps')

 ! set variable markers, JET tokamak:
  
  Do i=1, n_jet
     If (phase_id_jet(i)==0) Then                 ! ELM-free
        isyma(i) = dot
     Else If (phase_id_jet(i)==1) Then            ! Giant ELM
        isyma(i) = solid_diamond
     Else                                         ! Small ELM
        isyma(i) = empty_dot
     End If
  End Do

  Call discrim_var4(n_jet,isyma,                                           &
       &            elip=elip_jet      , elnel=elnel_jet,                  &
       &            elpinj=elpinj_jet  , lseplima=lseplima_jet,            &
       &            l_elnel=0.0_wp     , h_elnel=2.5_wp,     n_elnel=5,    &
       &            l_elip=0.5_wp      , h_elip=2.0_wp,      n_elip=6,     &
       &            l_elpinj=1.0_wp    , h_elpinj=3.0_wp,    n_elpinj=4,   &
       &            l_lseplima=-5.0_wp , h_lseplima=-2.0_wp, n_lseplima=6, &
       &            nd_le_to_ho=1     , nd_le_to_ve=1,                     &
       &            nd_le_bo_ho=1     , nd_le_bo_ve=1,                     &
       &            nd_ri_to_ho=1     , nd_ri_to_ve=1,                     &
       &            nd_ri_bo_ho=2     , nd_ri_bo_ve=1)

 ! producing postscript files:
! Call komplot(50,'discrim1_jet.eps')
  Call komplot(40,'discrim1_jet.ps')

 ! set variable markers, JFT-2M tokamak:

  Do i=1, n_jft
     If (phase_id_jft(i)==0) Then                 ! ELM-free
        isyma(i) = dot
     Else If (phase_id_jft(i)==1) Then            ! Giant ELM
        isyma(i) = solid_diamond
     Else                                         ! Small ELM
        isyma(i) = empty_dot
     End If
  End Do
 
  Call discrim_var4(n_jft,isyma,                                               &
       &            elip=elip_jft        , elnel=elnel_jft,                    &
       &            elpinj=elpinj_jft    , lseplima=lseplima_jft,              &
       &            l_elnel=0.0_wp       , h_elnel=2.5_wp,      n_elnel=5,     &
       &            l_elip=-2.5_wp       , h_elip=-1.0_wp,      n_elip=6,      &
       &            l_elpinj=-2.5_wp     , h_elpinj=1.0_wp,     n_elpinj=7,    &
       &            l_lseplima=-2.25_wp  , h_lseplima=-0.75_wp, n_lseplima=6,  &
       &            nd_le_to_ho=1        , nd_le_to_ve=1,                      &
       &            nd_le_bo_ho=1        , nd_le_bo_ve=1,                      &
       &            nd_ri_to_ho=2        , nd_ri_to_ve=1,                      &
       &            nd_ri_bo_ho=1        , nd_ri_bo_ve=1)

  ! producing postscript files:
!  Call komplot(50,'discrim1_jft.eps')
   Call komplot(40,'discrim1_jft.ps')

  ! Wandering write statement for debugging purpose:
   If (debug) Then
      Write(out,*) " ip(1:3)= ", ip(1:3)
      Write(out,*) " The program has been stopped regularly,"
      Write(out,*) " after ... making first postscript file "
      Stop
   End If

End Program chap5
!===============================================================================
!===============================================================================








