/* Program: conf4.sas                 */
/* Author:  ojk/ipp                   */
/* Date:    may 2004                  */
/* Update:                            */
/* Input file: conf4.csv (ascii)      */
/* Purpose: confinement data analysis */
/*          using SAS INSIGHT         */
/*          conclusions:              */

libname c '.';

options nocenter linesize=80;

proc import datafile="./conf4.csv" dbms=csv out=conf4 replace;
run;

*BEGIN data transformations;
data conf4_ext;
 set conf4;
 *density:;
  elnel      = log(enel);
  elnev      = log(enev);
  elne0      = log(ene0);
  elnel2     = elnel*elnel;
  elnel3     = elnel*elnel*elnel;
 *density peaking:;
  gn         = 0.5*(ene0/enel+enel/enev);
  gn0        = (ene0/enel);
  gnv        = (enel/enev);
  lgn0       = log(gn0);
  lgnv       = log(gnv);
  lgn1       = lgn0; *synonym;
  lgn2       = lgnv; *synonym;
  lgn        = log(gn);
 *shape:;
  leps       = log(amin/rgeo);
  lkappaa    = log(kappaa);
  ldeltap    = log(1+delta);
  lndeltap   = elnel*ldeltap;
  q95        = sh*qcyl;
  lq95       = log(q95);
  lqcyl      = log(qcyl);
  lqeng      = lqcyl;     *synonym;
  lshape     = log(sh);
  lFq        = lshape;    *synonym;
  lq95s      = lq95-1.0;
  lq95_2s    = lq95s*lq95s;
 *physical (dimensionless):;
  llarmorion = log(elarmorion);
  lnustar    = log(enustar);
 *engineering:;
  elngre20   = log(ngre20);
  elplth     = log(eplth);
  elenbi     = log(eenbi);
  elnela     = elnel+log(amin);
 * units for FIRE*;
  filnela    = elnel    - log(50) + log(amin/0.595);
  filenbi    = elenbi   - log(1.0);
  filplth    = elplth   - log(34.0);
  filnustar  = lnustar  - log(0.64);
  filngre20  = elngre20 - log(0.7);
  fillarmorion = llarmorion - log(1.2);
  fileps     = leps    - log(0.595/2.14);
  filkappaa  = lkappaa - log(1.85);
  filqcyl    = lqcyl   - log(2.0);
 *possible data selection!;  *notice;
  if (ind=0);
run;
*END data transformations;

*BEGIN medium and high density dataset;
data conf4_hine_ext;
 set conf4_ext;
  if elnel > log(5.0);
 *if elngre20 > -0.7;
run;
*END medium and high density dataset;

*BEGIN aug only;
data conf4_aug;
 set conf4_ext;
  if (tok='AUG');
run;
*END aug only;

*BEGIN standardise dataset;
proc standard data=conf4_ext out=conf4_ext_std std=1.0;
run;
*END standardise dataset;

* ad item a):;
proc insight;
 open conf4_ext;
 dist lgn0 lgnv;
 dist elngre20;
 scatter lgn0 * lgnv;
 fit lgn0 = llarmorion;
 fit lgnv = llarmorion;
 fit lgn0 = lnustar;
 fit lgnv = lnustar;
run;

*/************   * begin (un-)commented region; *(first column * un-comments);

* ad item b);
proc insight;
 open conf4_ext;
 fit lgn0 = leps lkappaa;
 fit lgn0 = leps lkappaa lshape;
 fit lgn0 = leps lkappaa ldeltap;
 fit lgnv = llarmorion lnustar;
 fit lgn0 = elplth elnela elenbi;
run;

* item c):;
proc insight;
 open conf4_ext;
 fit lgnv = llarmorion elngre20;
run;

proc insight;
 * 'with unstandardised variables:';
open conf4_ext;
 fit lgn0 lgnv = elnela elenbi elplth lnustar llarmorion leps lqcyl lkappaa;
 fit lgn0 lgnv = elnela elenbi elplth elngre20 llarmorion leps lqcyl lkappaa;
 * 'with standardised variables:';
 open conf4_ext_std;
 fit lgn0 lgnv = elnela elenbi elplth lnustar llarmorion leps lqcyl lkappaa;
 fit lgn0 lgnv = elnela elenbi elplth elngre20 llarmorion leps lqcyl lkappaa;
run;

************/  *end (un-)commented region;

title2 'eight variables, correlation matrix';
proc princomp data=conf4_ext;
 var elnela elenbi elplth lnustar llarmorion leps lqcyl lkappaa;
run;
proc princomp data=conf4_ext;
 var elnela elenbi elplth elngre20 llarmorion leps lqcyl lkappaa;
run;
title2 'nine variables, correlation matrix';
proc princomp data=conf4_ext;
 var elnela elenbi elplth lnustar elngre20 llarmorion leps lqcyl lkappaa;
run;

title2 'nine variables, covariance matrix';
proc princomp data=conf4_ext cov;
 var elnela elenbi elplth lnustar elngre20 llarmorion leps lqcyl lkappaa;
run;

title2;  * clears title2;

* ad item f):;
*BEGIN regression analysis based on principal components
 (because the condition of the dataset for 9 or 10 variables
  is just very marginal);

 %let sdlgn0         = 0.10;
 %let sdlgnv         = 0.10;
 %let sdelnela       = 0.02;
 %let sdelenbi       = 0.05;
 %let sdelplth       = 0.05;
 %let sdlnustar      = 0.20;
 %let sdelngre20     = 0.04;
 %let sdleps         = 0.04;
 %let sdllarmorion   = 0.10;
 %let sdlqcyl        = 0.05;
 %let sdlkappaa      = 0.04;
 %let sdlshape       = 0.08;

data conf4_exts; *s stands for standardised;
 set  conf4_ext;
  lgn0n         = lgn0     / (&sdlgn0);
  lgnvn         = lgnv     / (&sdlgnv);
  elnelan       = elnela   / (&sdelnela);
  elenbin       = elenbi   / (&sdelenbi);
  elplthn       = elplth   / (&sdelplth);
  lnustarn      = lnustar  / (&sdlnustar);
  elngre20n     = elngre20     / (&sdelngre20);
  lepsn         = leps         / (&sdleps);
  llarmorionn   = llarmorion   / (&sdllarmorion );
  lqcyln        = lqcyl        / (&sdlqcyl);
  lkappaan      = lkappaa      / (&sdlkappaa);
run;

proc princomp data=conf4_exts covariance outstat=pc1;
 var lgnvn elnelan elenbin elplthn lnustarn
     lepsn llarmorionn lqcyln lkappaan;
run;

proc princomp data=conf4_exts covariance outstat=pc2;
 var lgn0n elnelan elenbin elplthn lnustarn elngre20n
     lepsn llarmorionn lqcyln lkappaan;
run;

data pc1;
 set pc1;
  if _type_     = 'SCORE' and (_name_='Prin9'
                   or _name_ = 'Prin8' or _name_='Prin7');
  lgnvr         = -1.0*(&sdlgnv/&sdlgnv)*(lgnvn/lgnvn);
  elnelar       = -1.0*(&sdlgnv/&sdelnela)*(elnelan/lgnvn);
  elenbir       = -1.0*(&sdlgnv/&sdelenbi)*(elenbin/lgnvn);
  elplthr       = -1.0*(&sdlgnv/&sdelplth)*(elplthn/lgnvn);
  lnustarr      = -1.0*(&sdlgnv/&sdlnustar)*(lnustarn/lgnvn);
  lepsr         = -1.0*(&sdlgnv/&sdleps)*(lepsn/lgnvn);
  llarmorionr   = -1.0*(&sdlgnv/&sdllarmorion)*(llarmorionn/lgnvn);
  lqcylr        = -1.0*(&sdlgnv/&sdlqcyl)*(lqcyln/lgnvn);
  lkappaar      = -1.0*(&sdlgnv/&sdlkappaa)*(lkappaan/lgnvn);
  lfire = 0.085 + elnelar*log(50*0.595) + elenbir*log(1.0) + elplthr*log(34.0) +
          lnustarr*log(0.64)    + lepsr*log(0.595/2.14) +
          llarmorionr*log(1.2)  + lqcylr*log(2.0)       + lkappaar*log(1.85);
run;

proc means data=conf4_ext;
 var lgn0 lgnv;
run;

data pc2;
 set pc2;
  if _type_     = 'SCORE' and (_name_='Prin10'
                   or _name_ = 'Prin9' or _name_='Prin8');
  lgn0r         = -1.0*(&sdlgn0/&sdlgn0)*(lgn0n/lgn0n);
  elnelar       = -1.0*(&sdlgn0/&sdelnela)*(elnelan/lgn0n);
  elenbir       = -1.0*(&sdlgn0/&sdelenbi)*(elenbin/lgn0n);
  elplthr       = -1.0*(&sdlgn0/&sdelplth)*(elplthn/lgn0n);
  lnustarr      = -1.0*(&sdlgn0/&sdlnustar)*(lnustarn/lgn0n);
  elngre20r     = -1.0*(&sdlgn0/&sdelngre20)*(elngre20n/lgn0n);
  lepsr         = -1.0*(&sdlgn0/&sdleps)*(lepsn/lgn0n);
  llarmorionr   = -1.0*(&sdlgn0/&sdllarmorion)*(llarmorionn/lgn0n);
  lqcylr        = -1.0*(&sdlgn0/&sdlqcyl)*(lqcyln/lgn0n);
  lkappaar      = -1.0*(&sdlgn0/&sdlkappaa)*(lkappaan/lgn0n);
  lfire = 0.17 + elnelar*log(50*0.595) + elenbir*log(1.0) + elplthr*log(34.0) +
          lnustarr*log(0.64)    + elngre20r*log(0.7) + lepsr*log(0.595/2.14) +
          llarmorionr*log(1.2)  + lqcylr*log(2.0)    + lkappaar*log(1.85);
run;

*proc insight data=pc1;
*run;

title 'lgnv, nine variables';
proc print data=pc1;
run;

*proc insight data=pc2;
*run;

title 'lgn0, ten variables';
proc print data=pc2;
run;
*END regression analysis based on principal components;

*BEGIN OLS regression analysis;
proc reg data=conf4_ext;
  model lgn0 = lnustar;
  model lgnv = lnustar;
run;
proc reg data=conf4_aug;
  model lgn0 = lnustar;
  model lgnv = lnustar;
run;
proc reg data=conf4_ext;
 model lgnv  = elnela elenbi elplth lnustar
               leps llarmorion lqcyl lkappaa;
 model lgn0  = elnela elenbi elplth lnustar elngre20
               leps llarmorion lqcyl lkappaa;
run;
proc reg data=conf4_hine_ext;
 model lgnv  = elnela elenbi elplth lnustar
               leps llarmorion lqcyl lkappaa;
 model lgn0  = elnela elenbi elplth lnustar elngre20
               leps llarmorion lqcyl lkappaa;
run;
proc reg data=conf4_ext; *prediction for FIRE*:;
 model lgnv  = filnela filenbi filplth filnustar
               fileps fillarmorion filqcyl filkappaa;
 model lgn0  = filnela filenbi filplth filnustar filngre20
               fileps fillarmorion filqcyl filkappaa;
run;
quit;   *definitely exits proc reg;
*Remark: According to this first and rough model, a somewhat hollow central
         density profile is predicted for FIRE*;
*        possibly this might be offset when alpha particles are created
         that heat the central region.;
*END OLS regression analysis;

*BEGIN make postscript file and show listing using the xemacs editor;
* x xemacs conf4.lst;
x a2ps conf4.lst -o conf4.lst.a.ps;
x gv conf4.lst.a.ps;
*END make postscript file and show listing using the xemacs editor;

********* end of program conf4.sas ********************************************;
