! File: chap2.f90
! Date: June 10, 2003
! Update: May, 2004
!         September, 2004 
! Author: O. Kardaun
!
! Purpose: 
! making illustrations for Chap. 2 of Classical Methods of Statistics
! (Springer-verlag, 2004)

! (C) 2004 O. Kardaun, free software, released under GNU general public license, 
!     Version 2, June 1991, see http://www.gnu.org/copyleft/gpl.html  

!     Utilises routines from public domain KOMPLOT library (Version 8.0),
!     authored by J. Kraak, and available on internet, see
!     http://www.rug.nl/rc/hpcv/visualisation/komplot

! set precision 
!===============================================================================
!------------------------------------------------------------------------------+
Module set_PRECISION

  Integer, Parameter :: wp = Kind(1.0e0) ! single
  ! Integer, Parameter :: wp = Kind(1.0d0) ! double
  Integer, Parameter :: sp = Kind(1.0e0)   ! single
  Integer, Parameter :: dp = Kind(1.0d0)   ! double
 
End Module set_PRECISION
!------------------------------------------------------------------------------+
!===============================================================================

! utility subroutines
!===============================================================================
!------------------------------------------------------------------------------+
Module Utilities
!
! The module contains the following procedures:
!
! normp
!

Implicit None

Contains

Subroutine NORMP(Z, P, Q, PDF)

!
!	Normal distribution probabilities accurate to 1.e-15.
!	Z = no. of standard deviations from the mean.
!	P, Q = probabilities to the left & right of Z.   P + Q = 1.
!       PDF = the probability density.
!
!       Based upon algorithm 5666 for the error function, from:
!       Hart, J.F. et al, 'Computer Approximations', Wiley 1968
!
!       Programmer: Alan Miller
!
!	Latest revision - 30 March 1986
!
!       Slightly adapted to F90/95 by O. Kardaun, Dec 2003
!

!
! Imported parameters:
!
Use set_PRECISION, Only: wp 

Implicit None

! Imported Arguments:

Real(wp), Intent(in) :: Z
Real(wp), Intent(out) :: P
Real(wp), Intent(out) :: Q
Real(wp), Intent(out) :: PDF

! Local parameters:

        Real(wp) :: P0, P1, P2, P3, P4, P5, P6
        Real(wp) :: Q0, Q1, Q2, Q3, Q4, Q5, Q6, Q7
        Real(wp) :: cutoff, root2pi
        Real(wp) :: EXPNTL, ZABS

!-End of subroutine header------------------------------------------------------

! Initialisation:

        P0 = 220.2068679123761D0;
        P1 = 221.2135961699311D0;
        P2 = 112.0792914978709D0;
        P3 =  33.91286607838300D0;
        P4 =   6.373962203531650D0;
        P5 =   0.7003830644436881D0;
        P6 =   0.3526249659989109D-01;
        Q0 = 440.4137358247522D0;
        Q1 = 793.8265125199484D0;
        Q2 = 637.3336333788311D0;
        Q3 = 296.5642487796737D0;
        Q4 =  86.78073220294608D0;
        Q5 =  16.06417757920695D0;
        Q6 =   1.755667163182642D0;
        Q7 =   0.8838834764831844D-1;

        CUTOFF = 7.071D0;
        ROOT2PI = 2.506628274631001D0;

!old version:

!        IMPLICIT DOUBLE PRECISION (A-H, O-Z)
!
!        DATA P0, P1, P2, P3, P4, P5, P6/220.20 68679 12376 1D0,  &
!     &	  221.21 35961 69931 1D0, 112.07 92914 97870 9D0,         &
!     &	  33.912 86607 83830 0D0, 6.3739 62203 53165 0D0,         &
!     &	  .70038 30644 43688 1D0, .35262 49659 98910 9D-01/,      &
!     &	  Q0, Q1, Q2, Q3, Q4, Q5, Q6, Q7/440.41 37358 24752 2D0,  &
!     &	  793.82 65125 19948 4D0, 637.33 36333 78831 1D0,         &
!     &	  296.56 42487 79673 7D0, 86.780 73220 29460 8D0,         &
!     &	  16.064 17757 92069 5D0, 1.7556 67163 18264 2D0,         &
!     &	  .88388 34764 83184 4D-1/,                               &
!     &	  CUTOFF/7.071D0/, ROOT2PI/2.5066 28274 63100 1D0/
!

ZABS = Abs(Z)
!
!	|Z| > 37.
!
If (ZABS .Gt. 37.D0) Then
  PDF = 0.D0
  If (Z .Gt. 0.D0) Then
    P = 1.D0
    Q = 0.D0
  Else
    P = 0.D0
    Q = 1.D0
  End If
  Return
End If
!
!	|Z| <= 37.
!
EXPNTL = Exp(-0.5D0*ZABS**2)
PDF = EXPNTL/ROOT2PI
!
!	|Z| < CUTOFF = 10/sqrt(2).
!
If (ZABS .Lt. CUTOFF) Then
  P = EXPNTL*((((((P6*ZABS + P5)*ZABS + P4)*ZABS + P3)*ZABS +                  &
    &		P2)*ZABS + P1)*ZABS + P0)/(((((((Q7*ZABS + Q6)*ZABS +          &
    &		Q5)*ZABS + Q4)*ZABS + Q3)*ZABS + Q2)*ZABS + Q1)*ZABS +         &
    &		Q0)
!
!	|Z| >= CUTOFF.
!
Else
  P = PDF/(ZABS + 1.D0/(ZABS + 2.D0/(ZABS + 3.D0/(ZABS + 4.D0/                 &
    &		(ZABS + 0.65D0)))))
End If
!
If (Z .Lt. 0.D0) Then
  Q = 1.D0 - P
Else
  Q = P
  P = 1.D0 - Q
End If

!RETURN
End Subroutine NORMP
!===============================================================================

End Module Utilities
!------------------------------------------------------------------------------+
!===============================================================================

! various plot subroutines for Chapter 2
!===============================================================================
!------------------------------------------------------------------------------+
Module Plots
!
! The module contains the following procedures:
!
! powerfun1, invtest1, power2
!
  Implicit None

Contains

! calculate power function
!===============================================================================
Subroutine powerfun1

! Subroutine: powerfun1.f90
! Date: Dec 12, 2003
! Update: May, 2004
! Author: O. Kardaun
!
! Purpose: plotting illustration of the power function 
! for a normal distribution, included in Classical Methods of Statistics,
! Springer, Chapter 2
!
! Needed: library KOMPLOT, module KOMPLOT_EXT 
!
! Imported procedures (uncomment if needed)
!
  Use KOMPLOT_EXT, Only: sec_axis    !, only: format_y_axisnumbers.f90
  Use Utilities, Only: normp
!
! Imported parameters:
!
  Use set_PRECISION, Only: wp,sp 

  Implicit None

  ! Scalars:
  Integer, Parameter :: line_length=80
  Integer, Parameter :: out=6, nmax=200
  Integer, Parameter :: ncurves=10, npoints=100
  Integer :: isym, i, j, iformat
  Real(wp), Parameter :: pi=3.1415926
  Real(wp)   :: xs, xl, xh, ys, yl, yh, height, angle 
  Real(wp) :: t, th0, th, p1, q1, p2, q2, p3, q3, z1, z2, z3, prob
  Real(wp) :: sf_an, glab_xpct, glab_ypct, xp, yp, u_ahalf
  Logical :: debug

  ! Arrays:

  Real x1(1:nmax), y1(1:nmax), x2(1:nmax), y2(1:nmax), x3(1:nmax), y3(1:nmax)
  Real x4(1:nmax), y4(1:nmax), x5(1:nmax), y5(1:nmax)
  Real xline1(1:nmax), xline2(1:nmax), yline1(1:nmax), yline2(1:nmax)

!- End of subroutine header --------------------------------------------------

  ! initialisation --- set debug flag:
  debug = .True.

  Call frame_style('scientific')

  ! general options:
  Call scale_dimensions(1.0_wp)
  Call scale_annotations(1.75_wp)
  Call page_dimensions(19.0_wp,24.0_wp)

  xs= 7.5_wp;  ys=5.0_wp; ! plot size (in cm)
  xl=-3.0_wp;  xh=4.0_wp;
  yl= 0.0_wp;  yh=1.0_wp;

  isym=17; ! black dot

  Call font_switch_Character('$')

  Call tick_distances(xs/7,ys/5)

  Call Format_y_axisnumbers (2,0)

  Call x_margins(2.0,2.0) ! default
  Call y_margins(2.0,2.0) ! default
  Call format_x_axisnumbers(0,0) ! 1 decimal place, no exponent of 10 
  Call format_y_axisnumbers(1,0) ! 1 decimal place, no exponent of 10 

  Call shift_frame_in_page(2.75_wp,15.0_wp)

  !  Call frame(xs,xl,xh,ys,yl,yh,'$Nj$A0))','P{|T-$Nj0$a| > t}',' ')
  !  Call legend_in_blanked_out_area(1)
  Call frame(xs,xl,xh,ys,yl,yh,'','','')

  sf_an=1.2_wp
  glab_xpct = 15.0_wp
  glab_ypct = 8.5_wp

  ! parameters for shifts of normal distribution and power function: 
  th0=0.8_wp;
  t = 1.65_wp;

  ! general parameters of alphanumeric label for plot:

  height = 0.28_wp*sf_an           
  angle  = 0.0_wp 
  xp=xl-glab_xpct*sf_an*(xh-xl)/100.0_wp  
  yp=yh+glab_ypct*sf_an*(yh-yl)/100.0_wp   
  Call gstrng (xp,yp,height,angle,'(a)')

  ! th0:
  Call GSTRNG(th0-0.1       ,-0.25,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-0.1+0.25  ,-0.3 ,0.3,0.0,'$E0$A')
  !th0 + t:
  Call GSTRNG(th0+t-0.2     ,-0.25,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0+t-0.2+0.25,-0.3 ,0.3,0.0,'$E0')
  Call GSTRNG(th0+t-0.2+0.50,-0.25,0.4,0.0,'+t$A')
  !th0 - t:
  Call GSTRNG(th0-t-0.2     ,-0.25,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-t-0.2+0.25,-0.3 ,0.3,0.0,'$E0')
  Call GSTRNG(th0-t-0.2+0.50,-0.25,0.4,0.0,'-t$A')

  ! Plot the CDF functions for a normal distribution:
  ! y1 and y2 are around th0
  ! y3 and y4 are shifted by an amount +t and -t
  ! the power function y5 is the sum of y4 and y5

  Do j=1,npoints
     x1(j) = -3.0_wp + (Real(j-1,wp)*7.0_wp)/(npoints-1)
     z1 = x1(j)-th0
     z2 = x1(j)-th0-t
     z3 = x1(j)-th0+t 
     Call normp(z1,p1,q1,prob)
     y1(j) = p1
     y2(j) = q1  
     Call normp(z2,p2,q2,prob)
     y3(j) = p2
     Call normp(z3,p3,q3,prob) 
     y4(j) = q3
     y5(j) = y3(j)+y4(j)
  End Do

  ! horizontal line where the pwoer equals 0.5:
  xline1(1) = -3.0_wp
  xline1(2) = +4.0_wp
  yline1(1) = +0.5_wp
  yline1(2) = +0.5_wp 

  ! markers on the x-axis:
  xline2(1) = th0 - t
  yline2(1) = -0.05_wp
  xline2(2) = th0 - t
  yline2(2) = +0.05_wp

  xline2(3) = th0 + t
  yline2(3) = -0.05_wp
  xline2(4) = th0 + t
  yline2(4) = +0.05_wp

  Call line_style_nr (1) ! dashed
  Call curve(' ', npoints, x1(1), y1(1))
  Call curve(' ', npoints, x1(1), y2(1))
  Call line_style_nr (0) ! normal
  Call curve(' ', npoints, x1(1), y3(1)) 
  Call curve(' ', npoints, x1(1), y4(1)) 
  Call curve(' ', npoints, x1(1), y5(1))
  Call line_style_nr (3) ! another dashed
  Call curve(' ',2,xline1(1),yline1(1))
  Call line_style_nr (0) ! normal 
  Call curve(' ',2,xline2(1),yline2(1)) 
  Call curve(' ',2,xline2(3),yline2(3)) 

  ! Calculate arrays for two PDF functions for a normal distribution:
  Do j=1,npoints
     x1(j) = -3.0_wp + (Real(j-1,wp)*7.0)/(npoints-1)
     z1 = x1(j)-th0  
     Call normp(z1,p1,q1,prob)
     y1(j) = prob
     z2 = x1(j)-th0-t/2.0_wp  
     Call normp(z2,p2,q2,prob)
     y2(j) = prob
  End Do

  ! Plot first PDF functions for a normal distribution:
  xl=-3.0_wp;  xh=4.0_wp;
  yl= 0.0_wp;  yh=0.5_wp;
  Call tick_distances(xs/7.0_wp,ys/5.0_wp)
  Call shift_frame_in_page(2.75_wp,7.8_wp)
  Call frame(xs,xl,xh,ys,yl,yh,' ',' ',' ')
  xp=xl-glab_xpct*sf_an*(xh-xl)/100.0_wp  
  yp=yh+glab_ypct*sf_an*(yh-yl)/100.0_wp   
  Call gstrng (xp,yp,height,angle,'(b)')
  ! th0:
  Call GSTRNG(th0-0.1        ,-0.12 ,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-0.1+0.25   ,-0.15 ,0.3,0.0,'$E0$A')
  !th0 + t:
  Call GSTRNG(th0+t-0.45     ,-0.12 ,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0+t-0.45+0.25,-0.15 ,0.3,0.0,'$E0')
  Call GSTRNG(th0+t-0.45+0.50,-0.12 ,0.4,0.0,'+u$A')
  Call GSTRNG(th0+t-0.45+1.10,-0.15 ,0.3,0.0,'$Ma$A/2')
  !th0 - t:
  Call GSTRNG(th0-t-0.45     ,-0.12 ,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-t-0.45+0.25,-0.15 ,0.3,0.0,'$E0')
  Call GSTRNG(th0-t-0.45+0.50,-0.12 ,0.4,0.0,'-u$A')
  Call GSTRNG(th0-t-0.45+1.10,-0.15 ,0.3,0.0,'$Ma$A/2')

  Call line_style_nr (0) ! normal
  Call curve(' ', npoints, x1(1), y1(1))

  ! right hand tail area is filled:
  u_ahalf=1.96_wp;
  xline2(1) = th0+u_ahalf 
  yline2(1) = 0.0_wp
  Do j = 2, npoints
     xline2(j) = th0+u_ahalf +                                                 &
 &              (Real(j-2,wp))*(4.0_wp-u_ahalf-th0)/(npoints-2)
     z1 = xline2(j)-th0  
     Call normp(z1,p1,q1,prob)
     yline2(j) = prob
  End Do;
  xline2(npoints) = xline2(npoints-1)
  yline2(npoints) = 0.0_wp

  ! left hand tail area is filled:
  u_ahalf=1.96_wp;
  xline1(1) = th0-u_ahalf 
  yline1(1) = 0.0_wp
  Do j = 2, npoints
     xline1(j) = th0-u_ahalf +                                                 &
 &               (Real(j-2,wp))*(-3.0+u_ahalf-th0)/(npoints-2)
     z1 = xline1(j)-th0  
     Call normp(z1,p1,q1,prob)
     yline1(j) = prob
  End Do;
  xline1(npoints) = xline1(npoints-1)
  yline1(npoints) = 0.0_wp

  If (debug) Then
     Print *, 'second plot'
     Print *, 'xline1,yline1:'
     Do j=1,npoints
        Print *, xline1(j),'   ',yline1(j)
     End Do
     Print *, '    '
     Print *, 'xline2,yline2:'
     Do j=1,npoints
        Print *, xline2(j),'   ',yline2(j)
     End Do
  End If

  Call ps_graph_attributes(0.02,0.05) ! linewidth and grey value, almost black
  Call ps_fill_area('',npoints,xline1(1),yline1(1)) ! left  hand 
  Call ps_fill_area('',npoints,xline2(1),yline2(1)) ! right hand
  Call ps_graph_attributes(0.02,0.0) ! linewidth and again black color

  ! Plot second PDF functions for a normal distribution:
  Call shift_frame_in_page(2.75_wp,0.8_wp)
  Call frame(xs,xl,xh,ys,yl,yh,' ',' ',' ')

  xp=xl-glab_xpct*sf_an*(xh-xl)/100.0_wp  
  yp=yh+glab_ypct*sf_an*(yh-yl)/100.0_wp   
  Call gstrng (xp,yp,height,angle,'(c)')
  ! th0:
  Call GSTRNG(th0-0.1     ,-0.12 ,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-0.1+0.25,-0.15  ,0.3,0.0,'$E0$A')
  !th0 + t:
  Call GSTRNG(th0+t-0.45     ,-0.12,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0+t-0.45+0.25,-0.15 ,0.3,0.0,'$E0')
  Call GSTRNG(th0+t-0.45+0.50,-0.12,0.4,0.0,'+u$A')
  Call GSTRNG(th0+t-0.45+1.10,-0.15,0.3,0.0,'$Ma$A/2')
  !th0 - t:
  Call GSTRNG(th0-t-0.45     ,-0.12,0.4,0.0,'$Mq$E')
  Call GSTRNG(th0-t-0.45+0.25,-0.15,0.3,0.0,'$E0')
  Call GSTRNG(th0-t-0.45+0.50,-0.12,0.4,0.0,'-u$A')
  Call GSTRNG(th0-t-0.45+1.10,-0.15,0.3,0.0,'$Ma$A/2')

  Call line_style_nr (0) ! normal
  Call curve(' ', npoints, x1(1), y2(1))

  ! right hand tail area is filled:
  u_ahalf=1.65_wp;
  xline2(1) = th0+u_ahalf; 
  yline2(1) = 0.0_wp;
  Do j = 2, npoints
     xline2(j) = th0+u_ahalf +                                                 &
 &               (Real(j-2,wp))*(4.0_wp-u_ahalf-th0)/(npoints-2)
     z1 = xline2(j)-th0 - t/2 ! see also above 
     Call normp(z1,p1,q1,prob)
     yline2(j) = prob
  End Do;
  xline2(npoints) = xline2(npoints-1)
  yline2(npoints) = 0.0_wp

  ! left hand tail area is filled:
  u_ahalf=1.96_wp;
  xline1(1) = th0-u_ahalf 
  yline1(1) = 0.0_wp
  Do j = 2, npoints
     xline1(j) = th0-u_ahalf + &
 &               (Real(j-2,wp))*(-3.0+u_ahalf-th0)/(npoints-2)
     z1 = xline1(j)-th0 - t/2 ! see also above  
     Call normp(z1,p1,q1,prob)
     yline1(j) = prob
  End Do;
  xline1(npoints) = xline1(npoints-1)
  yline1(npoints) = 0.0_wp

! Call ps_graph_attributes(0.02,0.25) ! linewidth and grey value
  Call ps_graph_attributes(0.02,0.05) ! linewidth and grey value, almost black
  Call ps_fill_area('',npoints,xline1(1),yline1(1)) ! left  hand 
  Call ps_fill_area('',npoints,xline2(1),yline2(1)) ! right hand

 If (debug) Then
     Print *, 'third plot'
     Print *, 'xline1,yline1:'
     Do j=1,npoints
        Print *, xline1(j),'   ',yline1(j)
     End Do
     Print *, '    '
     Print *, 'xline2,yline2:'
     Do j=1,npoints
        Print *, xline2(j),'   ',yline2(j)
     End Do
  End If

 ! producing postscript files:
  Call komplot(50,'powerfun1.eps')
  Call komplot(40,'powerfun1.ps')

  ! producing display (X-windows):
  ! Call komplot(5)

End Subroutine powerfun1
!===============================================================================

! inversion of test statistic
!===============================================================================
Subroutine invtest1
!
! Subroutine: invtest1.f90
! Date: Dec 20, 2003
! Update:
! Author: O. Kardaun 
!
! Purpose: plotting illustration of the inverse of the test statistic
! included in Springer, Chapter 2, Fig. 2.4
!
! Needed: library KOMPLOT, module KOMPLOT_EXT 
!
! Imported procedures (uncomment if needed)
!
  Use KOMPLOT_EXT, Only: sec_axis    !, only: format_y_axisnumbers.f90
!
! Imported parameters:
!
  Use set_PRECISION, Only: wp,sp 

  Implicit None

  ! Scalars:
  Integer, Parameter :: line_length=80
  Integer, Parameter :: out=6, nmax=200
  Integer, Parameter :: ncurves=10, npoints=100
  Integer :: isym, i, j, iformat
  Real(wp), Parameter :: pi=3.1415926
  Real(wp)   :: xs, xl, xh, ys, yl, yh, height, angle 
  Real(wp) :: t, th0, th, p1, q1, p2, q2, p3, q3, z1, z2, z3, prob
  Real(wp) :: sf_an, glab_xpct, glab_ypct, xp, yp, u_ahalf, eps
  Logical :: debug

  ! Arrays:

  Real x1(1:nmax), y1(1:nmax), x2(1:nmax), y2(1:nmax), x3(1:nmax), y3(1:nmax)
  Real x4(1:nmax), y4(1:nmax), x5(1:nmax), y5(1:nmax)
  Real xline1(1:nmax), xline2(1:nmax), yline1(1:nmax), yline2(1:nmax)
  Real xline3(1:nmax), xline4(1:nmax), yline3(1:nmax), yline4(1:nmax)
  Real xline5(1:nmax), xline6(1:nmax), yline5(1:nmax), yline6(1:nmax)

!- End of subroutine header ----------------------------------------------------

  ! initialisation --- set debug flag:
  debug = .True.

  Call frame_style('scientific')

  ! general options:
  Call scale_dimensions(1.0_wp)
  Call scale_annotations(1.75_wp)
  Call page_dimensions(18.0_wp,9.001_wp) ! x and y (in cm)

  xs=5.0_wp;   ys=5.0_wp; ! plot size (in cm)
  xl=0.0_wp;   xh=0.8_wp;
  yl=0.0_wp;   yh=0.5_wp;

  isym=17 ; ! black dot

  Call font_switch_Character('$')

  Call tick_distances(xs/4,ys/5)

  Call x_margins(2.0_wp,2.0_wp) ! default
  Call y_margins(2.0_wp,2.0_wp) ! default
!  Call format_x_axisnumbers(0,0) ! 1 decimal place, no exponent of 10 
!  Call format_y_axisnumbers(0,0) ! 1 decimal place, no exponent of 10 

! First plot:

  Call shift_frame_in_page(0.0_wp,0.0_wp)

  Call suppress_axisnumbers(1,1) ! x-axis and y-axis 
  Call frame(xs,xl,xh,ys,yl,yh,'','','')

  sf_an=1.2_wp
  glab_xpct = 15.0_wp
  glab_ypct = 8.5_wp
 
! to check plot area:
!  angle=0.0_wp
!  height= -0.25_wp ! in units of cm
!  eps=0.5_wp

! corners of plot area:
! Call gstrng(-2.0_wp,-2.0_wp, height,angle,'A') ! left lower
! Call gstrng(6.75_wp,-2.0_wp, height,angle,'B') ! right lower
! Call gstrng(-2.0_wp,6.75_wp, height,angle,'C') ! right upper
! Call gstrng(6.75_wp,6.75_wp, height,angle,'D') ! left upper

  ! general parameters of alphanumeric label for plot:

  height = 0.28_wp*sf_an !in data units          
  angle  = 0.0_wp 
  xp=xl-glab_xpct*sf_an*(xh-xl)/100.0_wp  
  yp=yh+glab_ypct*sf_an*(yh-yl)/100.0_wp   
  Call gstrng (xp,yp,height,angle,'(a)')

  ! vertical legend:
  ! th:
  Call GSTRNG(-0.15     ,0.2 ,0.4,0.0,'$Mq$A')
  ! th_L:
  Call GSTRNG(-0.15     ,0.05,0.4,0.0,'$Mq$A')
  Call GSTRNG(-0.15+0.07,0.03,0.3,0.0,'L')
  ! th_U:
  Call GSTRNG(-0.15     ,0.40,0.4,0.0,'$Mq$A')
  Call GSTRNG(-0.15+0.07,0.38,0.3,0.0,'H')

  ! horizontal legend:
  ! t:
  Call GSTRNG(0.5 , -0.07 ,0.4 , 0.0 , 't')
  ! t_L:
  Call GSTRNG(0.15 , -0.07 , 0.4 , 0.0 , 't')
  Call GSTRNG(0.15+0.05,-0.07-0.03,0.3,0.0,'L') 
  ! t_U:
  Call GSTRNG(0.7 , -0.07, 0.4, 0.0, 't')
  Call GSTRNG(0.7+0.05,-0.07-0.03,0.3,0.0,'U')

  ! case description:
  Call GSTRNG(0.20 , -0.20 ,0.4 , 0.0 , '(regular case)')

  ! Coordinates left curve:

  x1(1) = 0.07; x1(2) = 0.165; x1(3) = 0.41;  x1(4)=0.50
  y1(1) = 0.12; y1(2) = 0.28;  y1(3) = 0.445; y1(4)=0.46

  ! Coordinates right curve:

  x2(1) = 0.33; x2(2) = 0.41;  x2(3) = 0.69;  x2(4)=0.76
  y2(1) = 0.06; y2(2) = 0.075; y2(3) = 0.28;  y2(4)=0.38

  ! Coordinates interval (horizontal and vertical):
  xline1(1) = x1(2); xline1(2)=x2(3); 
  xline2(1) = x2(2); xline2(2)=x1(3);
  yline1(1) = y1(2); yline1(2)=y2(3); 
  yline2(1) = y2(2); yline2(2)=y1(3);

  Call line_style_nr (0) ! normal
  Call curve(' ', 4, x1(1), y1(1))
  Call curve(' ', 4, x2(1), y2(1))
  Call curve(' ', 2, xline1(1), yline1(1))
  Call curve(' ', 2, xline2(1), yline2(1))

  If (debug) Then
     Print *, 'first plot'
     Print *, 'x1,y1:'
     Do j=1,4
        Print *, x1(j),'   ',y1(j)
     End Do
     Print *, ' '
     Print *, 'x1,y1:'
     Do j=1,4
        Print *, x1(j),'   ',y1(j)
     End Do
  End If

  ! Second plot:

  Call shift_frame_in_page(9.0_wp,0.0_wp)

! temporarily:
!   Call gridlines(1,1)
!   Call extra_gridlines(9,9)

  ! Call suppress_axisnumbers(1,1) ! x-axis and y-axis 
  Call frame(xs,xl,xh,ys,yl,yh,'','','')

  ! alphanumeric label:
  Call gstrng (xp,yp,height,angle,'(b)')

  ! vertical legend:
  ! th:
  Call GSTRNG(-0.15     ,0.2 ,0.4,0.0,'$Mq$A')
  ! horizontal legend:
  ! t:
  Call GSTRNG(0.5 , -0.07 ,0.4 , 0.0 , 't')

  ! Coordinates left curve:
  x1(1) = 0.20; x1(2) = 0.25; x1(3) = 0.29; x1(4)=0.24; x1(5)=0.27; x1(6)=0.30  
  y1(1) = 0.10; y1(2) = 0.15; y1(3) = 0.20; y1(4)=0.28; y1(5)=0.36; y1(6)=0.40
  ! Coordinates right curve:
  x2(1:6) = x1(1:6)+0.35
  y2(1:6) = y1(1:6)
  ! case description:
  Call GSTRNG(0.20 , -0.20 ,0.4 , 0.0 , '(irregular case)')

  Call line_style_nr (0) ! normal
  Call curve(' ', 6, x1(1), y1(1))
  Call curve(' ', 6, x2(1), y2(1))

  ! Coordinates intervals:
  xline1(1) = 0.26; xline1(2)=xline1(1)
  yline1(1) = 0.1;  yline1(2)=0.16            ! was 0.15
 
  xline2(1) = 0.26; xline2(2)=xline2(1)
  yline2(1) = 0.25; yline2(2)=0.345

  xline3(1) = 0.22; xline3(2)=xline3(1)+0.35  ! was 0.22
  yline3(1) = 0.12; yline3(2)=yline3(1)       ! was 0.15
 
  xline4(1) = 0.28; xline4(2)=xline4(1)+0.35
  yline4(1) = 0.19; yline4(2)=yline4(1)       ! was 0.20

  xline5(1) = 0.29;  xline5(2)=xline5(1)+0.35 ! was 0.28
  yline5(1) = 0.22;  yline5(2)=yline5(1)

  xline6(1) = 0.25;  xline6(2)=xline6(1)+0.35
  yline6(1) = 0.33;  yline6(2)=yline6(1)

! Plot intervals:
  Call curve(' ', 2, xline1(1), yline1(1))
  Call curve(' ', 2, xline2(1), yline2(1))
  Call curve(' ', 2, xline3(1), yline3(1))
  Call curve(' ', 2, xline4(1), yline4(1))
  Call curve(' ', 2, xline5(1), yline5(1))
  Call curve(' ', 2, xline6(1), yline6(1))

  ! producing postscript files:
  Call komplot(50,'invtest1.eps')
  Call komplot(40,'invtest1.ps')

  ! producing display (X-windows):
  ! Call komplot(5)

End Subroutine invtest1
!===============================================================================

! power function for Cauchy location problem
!===============================================================================
Subroutine power2
! File: power2.f90
! Date: May 19, 2003
! Update:
! Author: O. Kardaun
!
! Purpose: plotting ROC plot (on logit scale)
! of (1-beta) versus alpha for Chauchy distribution
!
! This version is for accept/reject region as given by the
! likelihood ratio statistic for one-sided testing
! 
! Needed: library KOMPLOT, module KOMPLOT_EXT 
!
! Imported procedures (uncomment if needed)
!
  Use KOMPLOT_EXT, Only: sec_axis      !, only: format_y_axisnumbers.f90
!
! Imported parameters:
!
  Use set_PRECISION, Only: wp,sp 

  Implicit None

  ! Scalars:
  Integer, Parameter :: line_length=80
  Integer, Parameter :: out=6, nmax=100
  Integer, Parameter :: ncurves1=10, ncurves2=8, npoints=80
  Integer, Parameter :: npointsp1=npoints+1
  Integer :: isym, i, j, k, iformat
  Real(wp), Parameter :: pi=3.1415926
  Real(wp)   :: xs, xl, xh, ys, yl, yh, height, angle 
  Real(wp)   :: lbt1,lbt2,lbth,t1,t2,th,alpha,beta

  ! Arrays:

  Real x(1:nmax), y(1:nmax)

!- End of subroutine header ----------------------------------------------------

  Call frame_style('scientific')

  ! general options:
  Call scale_dimensions(1.0)
  Call scale_annotations(1.75)
  Call page_dimensions(19.0,24.0)
  Call change(41,10.0,200.0) !max nr of frames per plots and of graphs per frame

  xs=11.0_wp;  ys=17.5_wp; ! plot size (in cm)  ! was 18.0 before Aug 31, 2004
  xl=-6.0_wp;  xh=6.0_wp;
  yl=-6.0_wp;  yh=6.0_wp;

  isym=17 ; ! black dot

  Call font_switch_CHARACTER('$')

  Call tick_distances(xs/6,ys/6)

  Call FORMAT_y_axisnumbers (2,0)

  Call legend_position(3.0,16.0)
  Call x_margins(2.0,2.0) ! default
  Call y_margins(2.0,2.0) ! default
  Call FORMAT_x_axisnumbers(1,0) ! 1 decimal place, no exponent of 10 
  Call FORMAT_y_axisnumbers(1,0) ! 1 decimal place, no exponent of 10 

  Call shift_frame_in_page(0.25_wp,0.0_wp)

  Call frame(xs,xl,xh,ys,yl,yh,'ln($Ma$A/(1-$Ma$A))','ln((1-$Mb$A)/$Mb$A)',' ')

  ! OJK: The following does not work for logit functions, since
  ! OJK: in komplot_ext_mod, it is only implemented for logarithmic functions, 
  ! OJK: at present, OJK, May 2003:
  !  Call sec_axis('h',xl,xh,yl,yh,'error of the first kind', &
  !  &             num_dens=2, num_fmt=202,debug=.false.)
  ! 
  !  Call sec_axis('v',xl,xh,yl,yh,'1 minus error of the second kind', &
  !  &             num_dens=2, num_fmt=202,debug=.false.)

  ! initialise:
  x=0.0_wp; y=0.0_wp;

  ! plot the line y=x:
  Do j=1,npoints
     x(j)=-6.0_wp+(12.0_wp*Real(j,wp)/Real(npoints,wp))
     y(j)=x(j)
  End Do
  Call curve(' ', npoints, x(1), y(1))
  ! plot the line y=-x:
  Do j=1,npointsp1
     x(j)=-6.0_wp+(12.0_wp*Real(j-1,wp)/Real(npoints,wp))
     y(j)=-1.0_wp*x(j)
  End Do
  Call curve(' ', npointsp1, x(1), y(1))

  ! I. plot the ROC curves on logit scale for shifted Cauchy distributions:

  Do i=1,ncurves1
     lbth=-5.0_wp+Real(i,wp)
     Do j=1,ncurves2
        lbt2=lbth+(8.0/Real(ncurves2,wp))*Real(j,wp)
  !         IF (lbt1 >= lbt2) cycle
        Do k=0,npoints
           lbt1=lbth-0.00001_wp-5.0_wp+(8.0_wp/Real(npoints,wp)) * Real(k,wp)
  !         IF (lbt1 >= lbt2) cycle
           th=2**lbth
           t1=2**lbt1
           t2=2**lbt2
           alpha=(1.0_wp/pi)*(Atan(Max(t1,t2))-Atan(Min(t1,t2)))
           beta=1.0_wp-(1.0_wp/pi)*(Atan(Max(t1,t2)-th))                       &
  &             +(1.0_wp/pi)*(Atan(Min(t1,t2)-th))
           If (k==0 .And. j==ncurves2 .And. th >= 1.0_wp) Then 
  ! plot the value of theta
              x(1)=Log(alpha/(1.0_wp-alpha))
              y(1)=Log((1-beta)/beta)
              height =0.25_wp
              angle  =0.0_wp
              iformat=201
              Call Greal(x(1),y(1),height,angle,iformat,th)
           Else
              x(k)=Log(alpha/(1.0_wp-alpha))
              y(k)=Log((1-beta)/beta)
           End If
        End Do !k
  !         CALL ps_graph_attributes(0.02,0.0) ! default width and grey value
        Call ps_graph_attributes(0.02_wp,0.05_wp*(ncurves2-j)) 
  !         CALL line_style_nr(ncurves2-j)
        Call curve(' ', npoints, x(1), y(1))
        Write(out,*) i,j,t1,t2,alpha,(x(k),k=1,3),(y(k),k=1,3)    
     End Do ! j
  End Do !i

  ! II. plot logit transformation on top (for horizontal axis)

  xs=11.0_wp;  ys=3.0_wp;
  xl=-6.0_wp;  xh=6.0_wp;
  yl=0.0_wp;   yh=1.0_wp;

  Call x_margins(2.0,2.0) ! default
  Call y_margins(1.0,0.5) ! was 0.5 was 0.0 before Aug 31, 2004
  Call FORMAT_y_axisnumbers(2,0) ! 2 decimal places, no exponent of 10 

  Call shift_frame_in_page(0.25_wp,19.3_wp)
  Call tick_distances(xs/6,ys/4)
  Call suppress_axisnumbers(1,0)      ! suppress x-axis numbers
  !  Call gridline_style_nr(0,12)     ! dotted grid lines
  Call ps_grid_attributes(0.01,0.1)   ! width and gray value of grid lines
  Call ps_frame_attributes(0.02,0.0)  ! width and gray value 
                                      ! of frame lines (default)
  Call gridlines(0,1)                 ! at several y positions, in x-direction
  Call frame(xs,xl,xh,ys,yl,yh,' ','$Ma$A',' ')

  Do j=1,npoints
     x(j) = -6.0+12.0*Real(j,wp)/npoints
     y(j) = Exp( x(j) ) / ( 1.0 + Exp(x(j)) )
  End Do
  Call curve(' ', npoints, x(1), y(1))

  ! plot an additional line for alpha = 5%:
  Call line_style_nr(2)
  Call PS_graph_attributes(0.01,0.3); ! LW and grey value, default: 0.02 and 0.0
  x(1) = -6.0; y(1)=0.05;
  x(2) = +6.0; y(2)=0.05;
  Call curve (' ',2,x(1),y(1))
  Call PS_graph_attributes(0.02,0.0); ! LW and grey value, default
  Call line_style_nr(0)               ! default 

  ! III. plot logit transformation on right (for vertical axis)

  xs = 4.75_wp; ys = 17.5_wp;
  xl = 0.0_wp;  xh = 1.0_wp;
  yl =-6.0_wp;  yh = 8.0_wp;

  Call x_margins(0.0,0.0)
  Call y_margins(2.0,2.0)        ! default
  Call format_x_axisnumbers(2,0) ! 2 decimal places, no exponent of 10 

  !  Call frame_style('GRID')
  Call shift_frame_in_page(13.75_wp,0.0_wp)
  Call tick_distances(xs/4,ys/7)
  Call suppress_axisnumbers(0,1) ! suppress y-axis numbers
  Call gridlines(1,0)            ! at several x positions, in y-direction
  Call frame(xs,xl,xh,ys,yl,yh,'1-$Mb$A',' ',' ')

  Do j=1,npoints
     x(j) = -0.01_wp+0.999_wp*Real(j,wp)/npoints
     y(j) = Log ( x(j) / (1.0_wp - x(j)) )
  End Do
  Call curve(' ', npoints, x(1), y(1))


  ! producing postscript files:
  Call komplot(50,'power2.eps')
  Call komplot(40,'power2.ps')


  ! producing display (X-windows):
  ! Call komplot(5)

End Subroutine power2
!===============================================================================

End Module Plots
!------------------------------------------------------------------------------+
!===============================================================================
!

! Main F-90 program; plots of Classical Methods of Statistics, Chapter 2 
Program chap2
!===============================================================================
!===============================================================================
!
! Note: procedure calls for different plots should be 
!  activated by uncommenting, and de-activated by commenting,
!  the relevant lines below
!
! Imported procedures (uncomment if needed)
! Use KOMPLOT_EXT, only: format_y_axisnumbers.f90
!
  Use Plots, Only: powerfun1, power2, invtest1
!
  Implicit None
!
!- End of header (program chap2)------------------------------------------------
!

! Call powerfun1

  Call power2

! Call invtest1
!
End Program chap2
!===============================================================================
!===============================================================================
