! File: chap3.f90
! Date: June 10, 2003
! Update: May, 2004
!         September, 2004
! Author: O. Kardaun
!
! Purpose: 
! making illustrations for Chap. 3 of Classical Methods of Statistics
! (Springer-Verlag, 2004)

! (C) 2004 O. Kardaun, free software, released under GNU general public license, 
!     Version 2, June 1991, see http://www.gnu.org/copyleft/gpl.html  

!     Utilises routines from public domain KOMPLOT library (Version 8.0),
!     authored by J. Kraak, and available on internet, see
!     http://www.rug.nl/rc/hpcv/visualisation/komplot

! set precision
!===============================================================================
!------------------------------------------------------------------------------+
Module set_PRECISION
!
      Integer, Parameter :: wp = Kind (1.0e0)! working prec., single
     ! Integer, Parameter :: wp = Kind (1.0d0) ! working prec., double
      Integer, Parameter :: sp = Kind (1.0e0)! single prec.
      Integer, Parameter :: dp = Kind (1.0d0)! double prec.
!
End Module set_PRECISION
!------------------------------------------------------------------------------+
!===============================================================================

! calculate probability densities for various distributions
!===============================================================================
!------------------------------------------------------------------------------+
Module ProbDist
!
      Implicit None
!
Contains
!
      Subroutine NORMP (Z, P, Q, PDF)
!
    !
    !	Normal distribution probabilities accurate to 1.0e-15.
    !	Z = no. of standard deviations from the mean.
    !	P, Q = probabilities to the left & right of Z.   P + Q = 1.
    !       PDF = the probability density.
    !
    !       Based upon algorithm 5666 for the error function, from:
    !       Hart, J.F. et al, 'Computer Approximations', Wiley 1968
    !
    !       Programmer: Alan Miller
    !
    !       Latest revision - 30 March 1986
    !
    !       Slightly adapted to F90/95 by O. Kardaun, Dec 2003
    !
!
         Use set_PRECISION, Only: dp
!
         Implicit None
!
    ! Imported Arguments:
!
         Real (dp), Intent (In) :: Z
         Real (dp), Intent (Out) :: P
         Real (dp), Intent (Out) :: Q
         Real (dp), Intent (Out) :: PDF
!
    ! Local parameters:
!
         Real (dp) :: P0, P1, P2, P3, P4, P5, P6
         Real (dp) :: Q0, Q1, Q2, Q3, Q4, Q5, Q6, Q7
         Real (dp) :: cutoff, root2pi
         Real (dp) :: EXPNTL, ZABS
!
!-End of subroutine header------------------------------------------------------
!
    ! Initialisation:
!
         P0 = 220.2068679123761D0
         P1 = 221.2135961699311D0
         P2 = 112.0792914978709D0
         P3 = 33.91286607838300D0
         P4 = 6.373962203531650D0
         P5 = 0.7003830644436881D0
         P6 = 0.3526249659989109D-01
         Q0 = 440.4137358247522D0
         Q1 = 793.8265125199484D0
         Q2 = 637.3336333788311D0
         Q3 = 296.5642487796737D0
         Q4 = 86.78073220294608D0
         Q5 = 16.06417757920695D0
         Q6 = 1.755667163182642D0
         Q7 = 0.8838834764831844D-1
!
         cutoff = 7.071D0
         root2pi = 2.506628274631001D0
!
    !old version:
!
    !        IMPLICIT DOUBLE PRECISION (A-H, O-Z)
    !
    !        DATA P0, P1, P2, P3, P4, P5, P6/220.20 68679 12376 1D0,      &
    !     &	  221.21 35961 69931 1D0, 112.07 92914 97870 9D0,         &
    !     &	  33.912 86607 83830 0D0, 6.3739 62203 53165 0D0,         &
    !     &	  .70038 30644 43688 1D0, .35262 49659 98910 9D-01/,      &
    !     &	  Q0, Q1, Q2, Q3, Q4, Q5, Q6, Q7/440.41 37358 24752 2D0,  &
    !     &	  793.82 65125 19948 4D0, 637.33 36333 78831 1D0,         &
    !     &	  296.56 42487 79673 7D0, 86.780 73220 29460 8D0,         &
    !     &	  16.064 17757 92069 5D0, 1.7556 67163 18264 2D0,         &
    !     &	  .88388 34764 83184 4D-1/,                               &
    !     &	  CUTOFF/7.071D0/, ROOT2PI/2.5066 28274 63100 1D0/
    !
!
         ZABS = Abs (Z)
    !
    !	|Z| > 37.
    !
         If (ZABS .Gt. 37.D0) Then
            PDF = 0.D0
            If (Z .Gt. 0.D0) Then
               P = 1.D0
               Q = 0.D0
            Else
               P = 0.D0
               Q = 1.D0
            End If
            Return
         End If
    !
    !	|Z| <= 37.
    !
         EXPNTL = Exp (-0.5D0*ZABS**2)
         PDF = EXPNTL / root2pi
    !
    !	|Z| < CUTOFF = 10/sqrt(2).
    !
         If (ZABS .Lt. cutoff) Then
            P = EXPNTL * ((((((P6*ZABS+P5)*ZABS+P4)*ZABS+P3)*ZABS+&
           & P2)*ZABS+P1)*ZABS+P0) / (((((((Q7*ZABS+Q6)*ZABS+Q5)*ZABS+&
           & Q4)*ZABS+Q3)*ZABS+Q2)*ZABS+Q1)*ZABS+Q0)
       !
       !	|Z| >= CUTOFF.
       !
         Else
            P = PDF / (ZABS+1.D0/(ZABS+2.D0/(ZABS+3.D0/(ZABS+4.D0/(ZABS+&
           & 0.65D0)))))
         End If
    !
         If (Z .Lt. 0.D0) Then
            Q = 1.D0 - P
         Else
            Q = P
            P = 1.D0 - Q
         End If
!
    !RETURN
      End Subroutine NORMP
!===============================================================================

! calculate Student's t distribution
!===============================================================================
      Subroutine tprob (x, df, pdens)
! Calculate the probability density function of Student's t
! distribution with df degrees of freedom
! using a simple analytic approximation for the beta function,
! which occurs as normalisation constant,
! see Sp. Ch 1.5 and the accuracy checking by program beta1.sas
! The relative accuracy better than some 1e-6 for 0.5<f<400, 
! which is more than sufficient for simple plots
!
    ! version 1, programmed by ojk at ipp
!
         Use set_PRECISION, Only: wp
!
         Implicit None
!
    ! Imported Arguments:
!
         Real (wp), Intent (In) :: x
         Integer, Intent (In) :: df
         Real (wp), Intent (Out) :: pdens
!
    ! Local parameters:
         Real (wp) :: f, fi, fih, lgamma_f, lgammar_f, gamma_f 
         Real (wp) :: lgamma_fh, lgammar_fh, gamma_fh, beta_h_fh
         Real (wp) :: lst2p
         Real (wp) :: densinv, pdensinv
         Real (wp), Parameter :: pi = 3.1415926_wp
         Integer, Parameter :: out = 6
         Logical :: debug
!
!-End of subroutine header------------------------------------------------------
!
         debug = .False.
!
         If (debug) Then
            Write (out,*) 'start of subroutine tprob:'
            Write (out,*) 'imported arguments:'
            Write (out,*) 'x, df'
            Write (out,*) x, df
         End If
!
        ! to convert integer into real and DIVIDE by 2.0
         f = Real (df, wp) / 2.0 
!
         If (debug) Then
            Write (out,*) 'start of subroutine tprob: '
            Write (out,*) 'imported arguments:'
            Write (out,*) 'x, df, f'
            Write (out,*) x, df, f
         End If
!
        ! calculate normalisation constant
         lst2p = 0.5_wp*(Log(2.0_wp*pi))
         fi=1.0_wp/f 
         lgammar_f = 0.001_wp*(-0.00220_wp+23.824_wp*fi*Log(1.0_wp+0.64_wp*fi) &
        & +  83.346_wp*fi-15.0105_wp*fi*fi+0.938715_wp*fi*fi*fi)               &
        & + 0.000001*( 2.42_wp*Cos(2.3_wp*pi/(f+0.25_wp))                      &
        &             -3.68_wp*Sin(2.3_wp*pi/(f+0.25_wp)) ) 
         lgamma_f = (f-0.5_wp) * Log (f) - f  + lst2p + lgammar_f
         gamma_f = Exp (lgamma_f)
         fih = 1.0_wp/(f+0.5_wp)
         lgammar_fh=0.001_wp*(-0.00220_wp+23.824_wp*fih*Log(1.0_wp+0.64_wp*fih)&
        & +  83.346_wp*fih-15.0105_wp*fih*fih+0.938715_wp*fih*fih*fih)         &
        & + 0.000001*( 2.42_wp*Cos(2.3_wp*pi/(f+0.75_wp))                      &
        &             -3.68_wp*Sin(2.3_wp*pi/(f+0.75_wp)) )
         lgamma_fh = f * Log (f+0.5_wp) - (f+0.5_wp) + lst2p + lgammar_fh 
         gamma_fh = Exp (lgamma_fh)
         beta_h_fh = Sqrt (pi) * gamma_f / gamma_fh
!
         If (debug) Then
            Print *, 'end of subroutine tprob: '
            Print *, 'intermediate arguments: '
            Print *, 'lgamma_f, gamma_f, lgamma_fh, gamma_fh, beta_h_fh&
           &'
            Print *, lgamma_f, gamma_f, lgamma_fh, gamma_fh, beta_h_fh
         End If
!
        ! calculate density
         densinv = (1.0_wp+(x*x)/(2.0_wp*f)) ** (f+0.5_wp)
         pdensinv = Sqrt (2.0_wp*f) * beta_h_fh * densinv
         pdens = 1.0_wp / pdensinv ! output parameter
!
         If (debug) Then
            Print *, ' End of subroutine tprob: '
            Print *, ' Exported arguments: '
            Print *, ' pdens'
            Print *, pdens
         End If
!
      End Subroutine tprob
!===============================================================================


! calculate F-probabilities
!===============================================================================
      Subroutine Fprob (x, df1, df2, pdens)
    ! calculates the probability density function of Snedecor's
    ! F-distribution with df degrees of freedom
    ! using a simple analytic approximation for the beta function,
    ! which occurs as normalisation constant,
    ! see Sp. Ch 1.5 and the checks by program beta1.sas
    ! relative accuracy better than some 1e-6, which is sufficient
    ! for simple plots 
!
    ! version 1, programmed by ojk at ipp
!
         Use set_PRECISION, Only: wp
!
         Implicit None
!
        ! Imported Arguments:
!
         Real (wp), Intent (In)  :: x
         Integer,   Intent (In)  :: df1, df2
         Real (wp), Intent (Out) :: pdens
!
        ! Local parameters:
         Real (wp) :: f, fi, lgamma_f, gamma_f, g, gi, lgamma_g, gamma_g,   &
        & fpg, fpgi, lgamma_fpg, gamma_fpg, lbeta_fg, beta_fg, c_fg
         Real (wp) :: lgammar_f, lgammar_g, lgammar_fpg
         Real (wp) :: lst2p
         Real (wp) :: ldens, dens
         Real (wp), Parameter :: pi = 3.1415926_wp
         Integer, Parameter :: out = 6
         Logical :: debug
!
!-End of subroutine header------------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .False.
!
         If (debug) Then
            Write (out,*) 'start of subroutine Fprob:'
            Write (out,*) 'imported arguments:'
            Write (out,*) 'x, df1, df2'
            Write (out,*) x, df1, df2
         End If
!
        ! to convert integer into real and DIVIDE by 2.0
         f = Real (df1, wp) / 2.0 
         g = Real (df2, wp) / 2.0 
         fpg = f + g
!
         If (debug) Then
            Write (out,*) 'start of subroutine Fprob: '
            Write (out,*) 'imported arguments:'
            Write (out,*) 'x, df1, df2, f, g'
            Write (out,*) x, df1, df2, f, g
         End If
!
        ! calculate normalisation constant
         lst2p = 0.5_wp*(Log(2.0_wp*pi)) ! log sqrt 2 pi 
         fi=1.0_wp/f 
         lgammar_f = 0.001_wp*(-0.00220_wp+23.824_wp*fi*Log(1.0_wp+0.64_wp*fi) &
        & + 83.346_wp*fi-15.0105_wp*fi*fi+0.938715_wp*fi*fi*fi)                &
        & + 0.000001*( 2.42_wp*Cos(2.3_wp*pi/(f+0.25_wp))                      &
        &             -3.68_wp*Sin(2.3_wp*pi/(f+0.25_wp)) )
         lgamma_f = (f-0.5_wp)*Log(f) - f + lst2p + lgammar_f
         gamma_f = Exp (lgamma_f)
         gi=1.0_wp/g
         lgammar_g = 0.001_wp*(-0.00220_wp+23.824_wp*gi*Log(1.0_wp+0.64_wp*gi) &
        & + 83.346_wp*gi-15.0105_wp*gi*gi+0.938715_wp*gi*gi*gi)                &
        & + 0.000001*( 2.42_wp*Cos(2.3_wp*pi/(g+0.25_wp))                      &
        &             -3.68_wp*Sin(2.3_wp*pi/(g+0.25_wp)) )
         lgamma_g = (g-0.5_wp)*Log(g) - g + lst2p + lgammar_g
         gamma_g = Exp (lgamma_g)
         fpgi=1.0_wp/(f+g)
         lgammar_fpg = 0.001_wp*( -0.00220_wp                                  &
        &                        +23.824_wp*fpgi*Log(1.0_wp+0.64_wp*fpgi)      &
        & + 83.346_wp*fpgi-15.0105_wp*fpgi**2.0_wp+0.938715_wp*fpgi**3.0_wp )  &
        & + 0.000001_wp*( 2.42_wp*Cos(2.3_wp*pi/(f+g+0.50_wp))                 &
        &                -3.68_wp*Sin(2.3_wp*pi/(f+g+0.50_wp)) )
         lgamma_fpg=(f+g-0.5_wp)*Log(f+g) - (f+g) + lst2p + lgammar_fpg
         gamma_fpg = Exp (lgamma_fpg)
!
         lbeta_fg = lgamma_f + lgamma_g - lgamma_fpg
         beta_fg = Exp (lbeta_fg)
         c_fg = (1.0_wp/beta_fg) * (f/g) ** (f)
!
         If (debug) Then
            Print *, ' End of subroutine Fprob: '
            Print *, ' Intermediate arguments: '
            Print *, ' lgamma_f, gamma_f, lgamma_g, gamma_g, beta_fg '
            Print *, lgamma_f, gamma_f, lgamma_g, gamma_g, beta_fg
            Print *, ' lgamma_fpg, gamma_fpg, c_fg '
            Print *, lgamma_fpg, gamma_fpg, c_fg
         End If
!
        ! calculate density
         ldens = (f-1.0_wp) * Log (x) - (f+g) * Log (1.0_wp + (f*x)/g)
         pdens = c_fg * Exp (ldens)
!
         If (debug) Then
            Print *, ' End of subroutine Fprob: '
            Print *, ' Exported arguments: '
            Print *, ' ldens, pdens '
            Print *, ldens, pdens
         End If
!
      End Subroutine Fprob
!===============================================================================

!
End Module ProbDist
!------------------------------------------------------------------------------+
!===============================================================================

! make various plots, Springer Chapter 3 
!-------------------------------------------------------------------------------
!===============================================================================
Module Plots
!
! The module contains the following procedures:
!
! arrow, bias1, ellip2, hyp, Fprob1, tprob1
!
      Implicit None
!
Contains
!
!===============================================================================
      Subroutine arrow (np, xx, yy, angle, length)
        ! plot a polyline with a single arrow
        ! OJK at IPP Garching, Jan 2004
        ! (version one)
        !
        !  Description:
        !  This subroutine uses common formulas from analytic geometry
        !  for lines in the form y=a0+a1*x and y=b0 - (1/a1) x, etc.,
        !  to calculate the tips of the arrows. For a1 equal to zero,
        !  a small eps is added to avoid infinity.
        !
        !  Another approach, not implemented in this first version,
        !  is to use polar coordinates instead of Cartesian coordinates.
        !
!
         Use set_PRECISION, Only: wp, sp, dp
         Implicit None
!
        !imported arguments
         Integer,   Intent (In) :: np
         Real (wp), Intent (In) :: xx (1:np), yy (1:np)
         Real (wp), Intent (In) :: angle, length
!
        ! meaning of arguments:
        ! np: number of points of the polyline
        ! xx: x coordinates of polyline
        ! yy: y coordinates of polyline
        ! angle: inner angle of arrow
        ! length: length of arrow (data units)
!
        ! local parameters:
!
         Integer, Parameter :: out = 6
         Integer :: i
         Real (wp) :: lambda, xm, ym, d, eps
         Real (wp) :: xa (3), ya (3)
         Logical :: debug
!
!- End of subroutine header ----------------------------------------------------
 
        ! initialisation --- set debug flag:
         debug = .True.
         eps = 0.0001_wp ! to avoid ill-conditioning for vertical lines
!
         lambda = length * Cos (angle)
         xm = xx (np) + lambda * (xx(np-1)-xx(np))
         ym = yy (np) + lambda * (yy(np-1)-yy(np))
         d = (xx(np-1)-xx(np)) / (yy(np-1)-yy(np)+eps)
         xa (1) = xm + lambda * Sin (angle) / Sqrt (1.0_wp+d*d)
         ya (1) = ym - d * (xa(1)-xm)
         xa (2) = xx (np)
         ya (2) = yy (np)
         xa (3) = xm - lambda * Sin (angle) / Sqrt (1.0_wp+d*d)
         ya (3) = ym - d * (xa(3)-xm)
!
         If (debug) Then
            Write (out,*) 'xx(*),yy(*)'
            Write (out,*) xx (1), yy (1)
            Write (out,*) xx (2), yy (2)
            Write (out,*) 'xm,ym,d,angle'
            Write (out,*) xm, ym, d, angle
            Write (out,*) 'xa(*),ya(*)'
            Do i = 1, 3
               Write (out,*) xa (i), ya (i)
            End Do
         End If
!
         Call polyline (' ', np, xx(1), yy(1))! polyline
         Call polyline (' ', 3, xa(1), ya(1))! arrow
!
      End Subroutine arrow
!===============================================================================

!
!===============================================================================
      Subroutine bias1
        ! calculates and plots pdf's of Student's t distribution,
        ! to illustrate a biased estimator, Springer, Ch3
        !
        Use set_PRECISION, Only: wp, sp
        Use ProbDist, Only: tprob
        !
         Implicit None
        !
        ! Scalars:
         Integer, Parameter :: out = 6, nmax = 200
         Integer, Parameter :: ncurves = 3, npoints = 100
         Integer :: isym, i, j, df1, df2
         Real (wp), Parameter :: pi = 3.1415926_wp
         Real (wp) :: xs, xl, xh, ys, yl, yh, h, height, angle
         Real (wp) :: t, th0, th, P1, Q1, P2, Q2, P3, Q3, z1, z2, z3, &
        & pdens
         Real (wp) :: sf_an, glab_xpct, glab_ypct, xp, yp, u_ahalf
         Logical :: debug
!
         Real (wp) m1, m2, s1, s2, x
         Real (wp) x1 (1:nmax), y1 (1:nmax), x2 (1:nmax), y2 (1:nmax)
         Real (wp) xline1 (1:nmax), xline2 (1:nmax), yline1 (1:nmax), &
        & yline2 (1:nmax)
!--- End of subroutine header --------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .True.
        !
         Call frame_style ('scientific')
        !
        ! general options:
         sf_an=1.5_wp
         Call scale_dimensions (1.0_wp)
         Call scale_annotations (sf_an)
         Call y_margins(2.0_wp,2.0_wp) ! standard
         Call x_margins(2.0_wp,0.5_wp) ! small margin left from graph 
         Call page_dimensions (15.0_wp, 12.0_wp)
        !
        ! plot size (in cm)
         xs = 10.0_wp
         ys =  7.0_wp
         xl = -4.0_wp
         xh =  4.0_wp
         yl =  0.0_wp
         yh =  0.5_wp
        !
         m1 =  0.0_wp ! mean
         m2 = -0.9_wp
!
         s1 = 3.0_wp ! standard deviation
         s2 = 1.0_wp
!
         df1 = 50 ! nearly standard normal
         df2 = 50
!
        ! Calculate arrays for some PDF functions for a t distribution:
         Do j = 1, npoints
            x1 (j) = - 4.0_wp + (Real(j-1, wp)*8.0_wp) / (npoints-1)
            x = (x1(j)-m1) / s1
            Call tprob (x, df1, pdens)
            y1 (j) = (1.0_wp/s1) * pdens
            x = (x1(j)-m2) / s2
            Call tprob (x, df2, pdens)
            y2 (j) = (1.0_wp/s2) * pdens
         End Do
!
!
         If (debug) Then
            Write (out,*) ' Some array elements in bias1.f90: '
            Write (out,*) ' x1(*),y1(*),y2(*) '
            Do j = 1, 5
               Write (out,*) x1 (j), y1 (j), y2 (j)
            End Do
         End If
!
!
        ! calculate two vertical lines at the mean values:
!
         xline1 (1) =  0.0_wp
         yline1 (1) =  0.0_wp
         xline1 (2) =  0.0_wp
         yline1 (2) =  0.395_wp / 3.0_wp
         xline2 (1) = -0.9_wp
         yline2 (1) =  0.0_wp
         xline2 (2) = -0.9_wp
         yline2 (2) =  0.395_wp
!
        ! plot:
!
         Call font_switch_CHARACTER ('$')
        !
         Call tick_distances (xs/8, ys/5)
         Call FORMAT_x_axisnumbers (0, 0)! 1 decimal place, no exponent of 10
         Call FORMAT_y_axisnumbers (1, 0)! 1 decimal place, no exponent of 10
        !
!        Call shift_frame_in_page (0.0_wp, 0.0_wp)
        !
         Call y_margins(2.0_wp,2.0_wp)
         Call ylab_above_y_axis (1)
         Call frame (xs, xl, xh, ys, yl, yh, '', '', '')
!
         h = 0.35_wp
         Call gstrng (-0.9-0.3, -0.11 , h, 0.0, 'E($Mq$A)')
         Call gstrng (-0.9+0.2, -0.085, h, 0.0, '^')
         Call gstrng (-0.1    , -0.11 , h, 0.0, '$Mq')
!
        !
         sf_an = 1.0_wp
        !
        ! make plot:
         Call line_style_nr (1)! dashed, unbiased
         Call curve (' ', npoints, x1(1), y1(1))
         Call line_style_nr (0)! normal, biased
         Call curve (' ', npoints, x1(1), y2(1))
         Call line_style_nr (1)! dashed, unbiased
         Call polyline (' ', 2, xline1(1), yline1(1))
         Call line_style_nr (0)! normal, biased
         Call polyline (' ', 2, xline2(1), yline2(1))
!
        ! producing postscript files:
         Call komplot (50, 'bias1.eps')
         Call komplot (40, 'bias1.ps')
!
      End Subroutine bias1
!===============================================================================

!
!===============================================================================
      Subroutine ellip2
        ! calculate an ellipse to illustrate an elliptical confidence region 
        ! for a parameter alpha
!
         Use set_PRECISION, Only: wp, sp, dp
         Implicit None
!
        ! Scalars:
         Integer, Parameter :: nmax = 40
         Integer, Parameter :: out = 6
         Integer :: isym, i, j
         Real (wp), Parameter :: pi = 3.1415926_wp
         Real (wp) :: xs, xl, xh, ys, yl, yh
         Real (wp) :: xm, ym, h, hs, sf_an
         Real (wp) :: k1, k2, s1, s2, m1, m2, ot, phix2, phiy2, rho
         Logical :: debug
!
        ! Arrays:
         Real (wp) :: x1 (1:nmax), y1 (1:nmax), x2 (1:nmax), y2 &
        & (1:nmax)
         Real (wp) :: z1 (1:nmax), z2 (1:nmax)
!
!- End of subroutine header ----------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .True.
!
         k1 = 1.0_wp
         s1 = 2.0_wp
         s2 = 3.0_wp
         m1 = 0.0_wp
         m2 = 0.0_wp
         phix2 = (3*pi) / 4.0_wp
         rho = Cos (phix2)
!
        ! fill array for original ellipse:;
         Do i = 1, nmax
            ot = (i/Real(nmax, wp)) * 2 * pi
            x1 (i) = m1 + k1 * s1 * Sin (ot)
            x2 (i) = m2 + k1 * s2 * Sin (ot+phix2)
         End Do
!
         xm = Sum (x1(1:nmax)) / Real (nmax, wp)
         ym = Sum (x2(1:nmax)) / Real (nmax, wp)
!
         If (debug) Then
            Write (out,*) ' Some array elements in ellip1.f90: '
            Write (out,*) ' x1(*),x2(*) '
            Do i = 1, nmax
               Write (out,*) x1 (i), x2 (i)
            End Do
         End If
!
         Call frame_style ('scientific')
!
         sf_an = 1.5_wp
        !    sf_an=1.0_wp
!
        ! general options:
         Call scale_dimensions (1.0_wp)
         Call shift_frame_in_page (0.0_wp, 0.0_wp)
         Call scale_annotations (sf_an)
!
!          Call scale_markers(0.50)
         Call page_dimensions (18.0_wp, 18.0_wp)
!
! plot size (in cm):
         xs = 10.0_wp
         ys = 10.0_wp
         xl = -4.0_wp
         xh =  4.0_wp
         yl = -4.0_wp
         yh =  4.0_wp
!
! black dot:
         isym = 17
!
         Call font_switch_CHARACTER ('$')
!
         Call tick_distances (xs/4.0_wp, ys/4.0_wp)
!
         Call FORMAT_x_axisnumbers (0, 0)
         Call FORMAT_y_axisnumbers (0, 0)
!
          ! CALL legend_position(3.0,16.0)
         Call frame (xs, xl, xh, ys, yl, yh, ' ', '', '')
         h  = 0.45 ! height in data units
         hs = 0.35 ! height of subscript
!
        ! legend along x-axis:
         Call gstrng (0.50, -5.0, h,  0.0, '$Ma$A')
         Call gstrng (0.85, -5.1, hs, 0.0, '$Aj')
!
        ! legend along y-axis:
         Call gstrng (-5.60, 2.6, h,  0.0, '$Ma$A')
         Call gstrng (-5.25, 2.5, hs, 0.0, 'j+1')
!
        ! plot:
         Call closed_curve (' ', nmax, x1(1), x2(1))
         Call markers (' ', 1, xm, ym, isym)
!
        ! producing postscript files:
         Call komplot (50, 'ellip2.eps')
         Call komplot (40, 'ellip2.ps')
!
      End Subroutine ellip2
!===============================================================================

!
!===============================================================================
      Subroutine hyp
        ! calculate an hyperbolic confidence band
!
         Use set_PRECISION, Only: wp, sp, dp
         Implicit None
!
        ! Scalars:
         Integer, Parameter :: nmax = 40
         Integer, Parameter :: out = 6
         Integer :: isym, i, j, np, np2
         Real (wp), Parameter :: pi = 3.1415926_wp
         Real (wp) :: angle, Len
         Real (wp) :: xs, xl, xh, ys, yl, yh
         Real (wp) :: xm, ym, ymp, h, hs, sf_an, rmse
         Real (wp) :: a0, a1, sxx, sxy
         Real (wp) :: k1, k2, s1, s2, m1, m2, ot, phix2, phiy2, rho
         Logical :: debug
!
          ! Arrays:
         Real (wp) :: x1 (1:nmax), y1 (1:nmax), x2 (1:nmax), y2 (1:nmax)
         Real (wp) :: x3 (1:nmax), y3 (1:nmax), x4 (1:nmax), y4 (1:nmax)
         Real (wp) :: x5 (1:nmax), y5 (1:nmax), x6 (1:nmax), y6 (1:nmax)
         Real (wp) :: z1 (1:nmax), z2 (1:nmax), z3 (1:nmax)
         Real (wp) :: z4 (1:nmax), z5 (1:nmax), z6 (1:nmax)
         Real (wp) :: xxsq (1:nmax), xysq (1:nmax), std_y (1:nmax)
         Real (wp) :: y_lower (1:nmax), y_upper (1:nmax)
!
!- End of subroutine header ----------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .True.
!
        ! original (artificial) data
         np = 11
         Do i = 1, np
            x1 (i) = Real (i, wp) + 3.0
         End Do
!
         y1 (1:np) = (/12.0, 9.0, 8.0, 11.0, 8.0, 9.0, 7.0, 6.0, 8.0, 4.0, 6.0/)
!
        ! average values:
         xm = Sum (x1(1:np)) / Real (np, wp)
         ym = Sum (y1(1:np)) / Real (np, wp)
!
        ! linear regression coefficients:
!
         xxsq (1:np) = (x1(1:np)-xm) ** 2
         xysq (1:np) = (x1(1:np)-xm) * (y1(1:np)-ym)
         sxx = Sum (xxsq(1:np))
         sxy = Sum (xysq(1:np))
         a1 = sxy / sxx
         a0 = ym - a1 * xm
!
         np2 = 30
        ! regression line:
         Do j = 1, np2
            x2 (j) = 3.0 + (16.0-3.0) * Real (j, wp) / Real (np2, wp)
            y2 (j) = a0 + a1 * x2 (j)
         End Do
!
        ! root mean squared error:
         rmse = Sqrt (Sum((y2(1:np)-y1(1:np))**2.0)/(Real(np, wp)-1.0))
!
        ! confidence band
         Do j = 1, np2
            x3 (j) = 3.0 + (16.0-3.0) * Real (j, wp) / Real (np2, wp)
            std_y (j) = rmse * Sqrt ((1.0/Real(np, &
           & wp))+(x3(j)-xm)**2.0/Sum(xxsq(1:np)))
            y_upper (j) = a0 + a1 * x3 (j) + std_y (j)
            y_lower (j) = a0 + a1 * x3 (j) - std_y (j)
         End Do
!
         ymp = a0 + a1 * xm
!
        ! x: right-low, right-up, left-up:
         z1 (1) = xm
         z1 (2) = xm
         z1 (3) = 2.0_wp
!
        ! y: right-low, right-up, left-up:
         z2 (1) = 0.0
         z2 (2) = ymp
         z2 (3) = ymp
!
        ! begin plotting graph:
!
         Call frame_style ('scientific')
         Call change(19, 1.0,0.0) ! to allow plotting outside axis border area

!
        ! general options:
         Call y_margins(2.0,0.5) ! small margin at top of graph
         Call x_margins(2.0,0.5) ! small margin left from graph 
         sf_an = 1.5_wp
         Call scale_dimensions (1.0)
         Call scale_annotations (sf_an)
         Call shift_frame_in_page (0.5_wp, 0.5_wp)
!
!          Call scale_markers(0.50)
         Call page_dimensions (15.0_wp, 15.0_wp) ! was xx,yy=18.0
!
        ! plot size (in cm)
         xs = 10.0_wp
         ys = 10.0_wp
         xl =  2.0_wp
         xh = 17.0_wp
         yl =  0.0_wp
         yh = 15.0_wp
!
        ! black dot
         isym = 17
!
         Call font_switch_CHARACTER ('$')
!
        ! nr of digits after dec. point, and exponent of 10:
!          Call FORMAT_x_axisnumbers (0.,0.)
!          Call FORMAT_y_axisnumbers (0.,0.)
         Call FORMAT_x_axisnumbers (0, 0)! 1 decimal place, no exponent of 10
         Call FORMAT_y_axisnumbers (0, 0)! 1 decimal place, no exponent of 10
!
         Call tick_distances (xs/5.0_wp, ys/5.0_wp)
         Call frame (xs, xl, xh, ys, yl, yh, '', '', '')
!
!
        ! legend along vertical axis:
         h  = 0.45_wp ! height in data units
         hs = 0.35_wp ! height of subscript
         Call gstrng (14.0,-1.8, h, 0.0, 'x')
         Call gstrng (-1.0, 13.0, h, 0.0, 'y')
!
         If (debug) Then
            Write (out,*) ' Some array elements in hyp.f90: '
            Write (out,*) ' x1(*),y1(*) '
            Do i = 1, np
               Write (out,*) x1 (i), y1 (i)
            End Do
            Write (out,*) 'np, isym, rmse'
            Write (out,*) np, isym, rmse
            Write (out,*) ' x2(*),y2(*) '
            Do i = 1, np
               Write (out,*) x2 (i), y2 (i)
            End Do
            Write (out,*) ' x3(*),std_y(*) '
            Do i = 1, np
               Write (out,*) x3 (i), y_upper (i)
            End Do
         End If
!
        ! plot data:
         Call markers (' ', np, x1(1), y1(1), isym)
!
        ! plot regression line:
         Call curve (' ', np2, x2(1), y2(1))
!
        ! confidence band:
         Call curve (' ', np2, x3(1), y_upper(1))
         Call curve (' ', np2, x3(1), y_lower(1))

        ! horizontal arrow:
         x4 (1) = 12.0_wp
         x4 (2) = 13.2_wp
         y4 (1) = - 1.5_wp
         y4 (2) = - 1.5_wp
         angle = (45.0/360.0) * 2 * pi
         Len = 0.3

        ! vertical arrow:
         x5 (1) =  -0.75_wp
         x5 (2) =  -0.75_wp
         y5 (1) =  10.4_wp
         y5 (2) =  11.6_wp
         angle  = (45.0_wp/360.0_wp) * 2.0_wp * pi
         Len    =   0.3_wp

!       ! draw arrows:
         Call arrow (2, x4(1), y4(1), angle, len)
         Call arrow (2, x5(1), y5(1), angle, len)
!
        ! legend for the mean values:
         Call gstrng (xm-0.3, -0.9, h, 0.0, '_')
         Call gstrng (xm-0.3, -1.8, h, 0.0, 'x')
         Call gstrng (-1.0, ym+0.7, h, 0.0, '_')
         Call gstrng (-1.0, ym-0.2, h, 0.0, 'y')
!
        ! plot (dashed) lines from the center of gravity to the x and y axes:
         Call line_style_nr (1)! dashed
         Call polyline (' ', 3, z1(1), z2(1))
!
        ! producing postscript files:
         Call komplot (50, 'hyp1new.eps')
         Call komplot (40, 'hyp1new.ps')
!
! Make a second plot to illustrate inverse regression:
!
        ! general options:
         Call frame_style ('scientific')
         Call y_margins(2.0_wp,0.5_wp) ! small margin at top of graph
         Call x_margins(2.0_wp,0.5_wp) ! small margin left from graph 
         Call page_dimensions (15.0_wp, 15.0_wp)

! plot size (in cm), same as above
         xs =  9.0_wp
         ys =  9.0_wp
         xl =  2.0_wp
         xh = 17.0_wp
         yl =  0.0_wp
         yh = 15.0_wp
!
!
         Call FORMAT_x_axisnumbers (0, 0)! 0 decimal place, no exponent of 10
         Call FORMAT_y_axisnumbers (0, 0)! 0 decimal place, no exponent of 10
!
         Call tick_distances (xs/5.0_wp, ys/5.0_wp)
!
         sf_an = 1.5_wp
         Call scale_dimensions (1.0_wp)
         Call scale_annotations (sf_an)
         Call shift_frame_in_page (0.5_wp, 0.0_wp)
!
         Call frame (xs, xl, xh, ys, yl, yh, '', '', '')
!
        ! legend along vertical axis:
         h  = 0.45_wp ! height in data units
         hs = 0.35_wp ! height of subscript
         Call gstrng (14.0, -1.8, h, 0.0, 'x')
         Call gstrng (-1.0, 13.0, h, 0.0, 'y')
!
        ! plot data:
         Call markers (' ', np, x1(1), y1(1), isym)
!
        ! plot regression line:
         Call curve (' ', np2, x2(1), y2(1))
!
        ! confidence band:
         Call curve (' ', np2, x3(1), y_upper(1))
         Call curve (' ', np2, x3(1), y_lower(1))
!
         x4 (1) = 12.0_wp
         x4 (2) = 13.2_wp
         y4 (1) = - 0.8_wp
         y4 (2) = - 0.8_wp
         angle = (45.0/360.0) * 2 * pi
         Len = 0.3
!
        ! x: right-low, right-up, left-up:
         z3 (1) = 9.2_wp
         z3 (2) = 9.2_wp
         z3 (3) = 2.0_wp
!
        ! y: right-low, right-up, left-up:
         z4 (1) = 0.0_wp
         z4 (2) = 8.67_wp
         z4 (3) = 8.67_wp
!
        ! x: right-low, right-up, left-up:
         z5 (1) = 5.9_wp
         z5 (2) = 5.9_wp
!
        ! y: right-low, right-up, left-up:
         z6 (1) = 0.0_wp
         z6 (2) = 8.67_wp
!
        ! plot (dashed) lines illustrating inverse regression:
         Call line_style_nr (1)! dashed
         Call polyline (' ', 3, z3(1), z4(1))
         Call polyline (' ', 2, z5(1), z6(1))
!
        ! producing postscript files:
         Call komplot (50, 'hyp2new.eps')
         Call komplot (40, 'hyp2new.ps')
!
!
      End Subroutine hyp
!===============================================================================

!
!===============================================================================
      Subroutine Fprob1
        ! calculate and plot probability density functions 
        ! of Fisher's F distribution
!
         Use set_PRECISION, Only: wp, sp
         Use ProbDist, Only: Fprob
!
         Implicit None
!
        ! Scalars:
         Integer, Parameter :: out = 6, nmax = 200
         Integer, Parameter :: ncurves = 3, npoints = 100
         Integer :: isym, i, j, df1, df2, df3, df4, df5
         Real (wp), Parameter :: pi = 3.1415926_wp
         Real (wp) :: xs, xl, xh, ys, yl, yh, height, angle
         Real (wp) :: t, th0, th, P1, Q1, P2, Q2, P3, Q3, z1, z2, z3, pdens
         Real (wp) :: sf_an, glab_xpct, glab_ypct, xp, yp, u_ahalf
         Logical :: debug
!
        ! Arrays:
!
         Real x1 (1:nmax), y1 (1:nmax), x2 (1:nmax), y2 (1:nmax)
         Real x3 (1:nmax), y3 (1:nmax)
         Real x4 (1:nmax), y4 (1:nmax), x5 (1:nmax), y5 (1:nmax)
         Real xline1 (1:nmax), xline2 (1:nmax), yline1 (1:nmax), yline2 (1:nmax)
         Real xline3 (1:nmax), xline4 (1:nmax), yline3 (1:nmax), yline4 (1:nmax)
         Real Fcrit (1:5)! critical values for the F-distribution
!
!- End of subroutine header ----------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .True.
!
         Call frame_style ('scientific')
!
         sf_an=1.5
        ! general options:
         Call scale_dimensions (1.0)
         Call scale_annotations (sf_an)
         Call page_dimensions (16.0, 10.0)
!
        ! plot size (in cm)
         xs = 10.0_wp
         ys =  7.0_wp
         xl =  0.0_wp
         xh =  4.0_wp
         yl =  0.0_wp
         yh =  1.0_wp
!
         df1 = 5
         df2 = 100
!
        ! insert critical values for F-distribution
        ! according to the above degrees-of-freedom !
         Fcrit (1) = 2.3_wp
!
        ! Calculate arrays for PDF functions of an F distribution:
         Do j = 1, npoints
            x1 (j) = 0.0_wp + (Real(j-1, wp)*4.0_wp) / (npoints-1)
            Call Fprob (x1(j), df1, df2, pdens)
            y1 (j) = pdens
         End Do
!
         If (debug) Then
            Print *, 'x1(*), y1(*)'
            Do j = 1, 10
               Print *, x1 (j), y1 (j)
            End Do
         End If
!
         Call font_switch_CHARACTER ('$')
!
         Call tick_distances (xs/8, ys/5)
         Call FORMAT_x_axisnumbers (1, 0)! 1 decimal place, no exponent of 10
         Call FORMAT_y_axisnumbers (1, 0)! 1 decimal place, no exponent of 10
!
         Call shift_frame_in_page (0.0_wp, 0.0_wp)
!
!         Call ylab_above_y_axis (1)
         Call x_margins(3.0,2.0)
         Call y_margins(1.5,0.5)
         Call frame (xs, xl, xh, ys, yl, yh, 'x', 'f(x)', '')
!         Call gstrng(-1.2, 0.75, 0.45, 0.0, 'f(x)')
!
         sf_an = 1.5
!
        ! make plot:
         Call line_style_nr (0)! normal
         Call curve (' ', npoints, x1(1), y1(1))
 !
        !------------------------
        ! tail areas :
        !------------------------
        ! right hand tail area is filled:
         xline1 (1) = Fcrit (1)
         yline1 (1) = 0.0_wp
         Do j = 2, npoints
            xline1 (j) = Fcrit (1) + (Real(j-2, wp)) * (4.0_wp-Fcrit(1)) / &
           & (npoints-2)
            Call Fprob (xline1(j), df1, df2, pdens)
            yline1 (j) = pdens
         End Do
         xline1 (npoints) = xline1 (npoints-1)
         yline1 (npoints) = 0.0_wp
!
!
         Call ps_graph_attributes (0.02_wp, 0.05_wp)! linewidth and grey value
         Call ps_fill_area ('', npoints, xline1(1), yline1(1))! right hand
!
        ! producing postscript file:
         Call komplot (50, 'Fprob1.eps')
         Call komplot (40, 'Fprob1.ps')
!
!
      End Subroutine Fprob1
!===============================================================================
!

!===============================================================================
      Subroutine tprob1
        ! calculate and plot probability density functions 
        ! of Student's t distribution,
        !
         Use set_PRECISION, Only: wp, sp
         Use ProbDist, Only: tprob
!
         Implicit None
!
          ! Scalars:
         Integer, Parameter :: out = 6, nmax = 200
         Integer, Parameter :: ncurves = 3, npoints = 100
         Integer :: isym, i, j, df1, df2, df3, df4, df5
         Real (wp), Parameter :: pi = 3.1415926_wp
         Real (wp) :: xs, xl, xh, ys, yl, yh, height, angle
         Real (wp) :: t, th0, th, P1, Q1, P2, Q2, P3, Q3, z1, z2, z3, &
        & pdens
         Real (wp) :: sf_an, glab_xpct, glab_ypct, xp, yp, u_ahalf
         Logical :: debug
!
          ! Arrays:
!
         Real x1 (1:nmax), y1 (1:nmax), x2 (1:nmax), y2 (1:nmax)
         Real x3 (1:nmax), y3 (1:nmax)
         Real x4 (1:nmax), y4 (1:nmax), x5 (1:nmax), y5 (1:nmax)
         Real xline1 (1:nmax), xline2 (1:nmax), yline1 (1:nmax), yline2 &
        & (1:nmax)
         Real xline3 (1:nmax), xline4 (1:nmax), yline3 (1:nmax), yline4 &
        & (1:nmax)
         Real tcrit (1:5)! critical values for the t-distribution
!
!- End of subroutine header ----------------------------------------------------
!
        ! initialisation --- set debug flag:
         debug = .False.
!
          Call frame_style ('scientific')
!
         sf_an = 1.5 
        ! general options:
         Call scale_dimensions (1.0)
         Call scale_annotations (sf_an)
         Call page_dimensions (16.0, 11.0)
!
        ! plot size (in cm)
         xs = 10.0_wp
         ys =  7.0_wp
         xl = -4.0_wp
         xh =  4.0_wp
         yl =  0.0_wp
         yh =  0.5_wp
!
         df1 = 1 ! 1 degree of freedom ! later perhaps in a loop
         df2 = 2
         df3 = 3
         df4 = 5
         df5 = 30
!
          ! insert critical values for t-distribution
          ! according to the above degrees-of-freedom !
         tcrit (1) = 12.7_wp
         tcrit (2) = 4.30_wp
         tcrit (3) = 3.18_wp
         tcrit (4) = 2.57_wp
         tcrit (5) = 2.00_wp
 !
        ! Calculate arrays for some PDF functions for a t distribution:
         Do j = 1, npoints
            x1 (j) = - 4.0_wp + (Real(j-1, wp)*8.0_wp) / (npoints-1)
            Call tprob (x1(j), df1, pdens)
            y1 (j) = pdens
            Call tprob (x1(j), df2, pdens)
            y2 (j) = pdens
            Call tprob (x1(j), df3, pdens)
            y3 (j) = pdens
            Call tprob (x1(j), df4, pdens)
            y4 (j) = pdens
            Call tprob (x1(j), df5, pdens)
            y5 (j) = pdens
         End Do
!
         If (debug) Then
            Print *, 'x1(*), y1(*)'
            Do j = 1, 10
               Print *, x1 (j), y1 (j)
            End Do
         End If
!
         Call font_switch_CHARACTER ('$')
!
         Call tick_distances (xs/8, ys/5)
         Call FORMAT_x_axisnumbers (0, 0)! 1 decimal place, no exponent of 10
         Call FORMAT_y_axisnumbers (1, 0)! 1 decimal place, no exponent of 10
!
         Call shift_frame_in_page (0.0_wp, 0.0_wp)
!
!         Call ylab_above_y_axis (1)
         Call x_margins(3.0_wp,2.0_wp)
         Call frame (xs, xl, xh, ys, yl, yh, 't', 'f(t)', '')
!         Call gstrng(-6.5, 0.35, 0.45, 0.0, 'f(t)')
!
         sf_an = 1.5_wp
!
        ! make plot:
         Call line_style_nr (0)! normal
         Call curve (' ', npoints, x1(1), y1(1))
         Call curve (' ', npoints, x1(1), y2(1))
         Call curve (' ', npoints, x1(1), y3(1))
         Call curve (' ', npoints, x1(1), y4(1))
         Call curve (' ', npoints, x1(1), y5(1))
!
        !------------------------
        ! tail areas for df3:
        !------------------------
        ! right hand tail area is filled:
         xline1 (1) = tcrit (3)
         yline1 (1) = 0.0_wp
         Do j = 2, npoints
            xline1 (j) = tcrit (3) + (Real(j-2, wp)) * (4.0_wp-tcrit(3)) / &
           & (npoints-2)
            Call tprob (xline1(j), df3, pdens)
            yline1 (j) = pdens
         End Do
         xline1 (npoints) = xline1 (npoints-1)
         yline1 (npoints) = 0.0_wp
!
          ! left hand tail area is filled:
         xline2 (1) = - 1.0 * tcrit (3)
         yline2 (1) = 0.0_wp
         Do j = 2, npoints
            xline2 (j) = - 1.0_wp * tcrit (3) + (Real(j-2, wp)) * &
           & (-4.0_wp+tcrit(3)) / (npoints-2)
            Call tprob (xline2(j), df3, pdens)
            yline2 (j) = pdens
         End Do
         xline2 (npoints) = xline2 (npoints-1)
         yline2 (npoints) = 0.0_wp
!
        !------------------------
        ! tail areas for df5:
        !------------------------
        ! right hand tail area is filled:
         xline3 (1) = tcrit (5)
         yline3 (1) = 0.0_wp
         Do j = 2, npoints
            xline3 (j) = tcrit (5) + (Real(j-2, wp)) * (4.0_wp-tcrit(5)) / &
           & (npoints-2)
            Call tprob (xline3(j), df5, pdens)
            yline3 (j) = pdens
         End Do
         xline3 (npoints) = xline3 (npoints-1)
         yline3 (npoints) = 0.0_wp
!
        ! left hand tail area is filled:
         xline4 (1) = - 1.0_wp * tcrit (5)
         yline4 (1) = 0.0_wp
         Do j = 2, npoints
            xline4 (j) = - 1.0_wp * tcrit (5) + (Real(j-2, wp)) * &
           & (-4.0_wp+tcrit(5)) / (npoints-2)
            Call tprob (xline4(j), df5, pdens)
            yline4 (j) = pdens
         End Do
         xline4 (npoints) = xline4 (npoints-1)
         yline4 (npoints) = 0.0_wp
!
         Call ps_graph_attributes (0.02, 0.05)! linewidth and grey value
         Call ps_fill_area ('', npoints, xline1(1), yline1(1))! left  hand
         Call ps_fill_area ('', npoints, xline2(1), yline2(1))! right hand
         Call ps_fill_area ('', npoints, xline3(1), yline3(1))! left  hand
         Call ps_fill_area ('', npoints, xline4(1), yline4(1))! right hand
!
        ! producing postscript file:
         Call komplot (50, 'tprob1.eps')
         Call komplot (40, 'tprob1.ps')
!
!
      End Subroutine tprob1
!===============================================================================
 
End Module plots
!------------------------------------------------------------------------------+
!===============================================================================

! Main F-90/95 program 
Program chap3
!===============================================================================
!===============================================================================
!
! Main program for plots of Chap. 3 of Classical Methods of Statistics
!
! Note: procedure calls for different plots should be 
!  activated by uncommenting and de-activated by commenting
!  the relevant lines below
!
! Imported Procedures (uncomment if needed)
! Use KOMPLOT_EXT, only: format_y_axisnumbers.f90 ! from komplot_ext_mod.f90
!

      Use Plots, Only: ellip2, hyp, tprob1, Fprob1, bias1
!
      Implicit None
!
!- End of header (program chap3)------------------------------------------------
      
!      Call ellip2

!      Call hyp

      Call tprob1

      Call Fprob1

!      Call bias1
!
End Program chap3
!===============================================================================
!===============================================================================

