! File: chap7.f90
! Date: June 10, 2003
! Update: May, 2004
!         September, 2004
! Author: O. Kardaun
!
! Purpose: 
! making illustrations for Chap. 7 of Classical Methods of Statistics
! (Springer-Verlag, 2004)

! (C) 2004 O. Kardaun, free software, released under GNU general public license, 
!     Version 2, June 1991, see http://www.gnu.org/copyleft/gpl.html  

!     Utilises routines from public domain KOMPLOT library (Version 8.0),
!     authored by J. Kraak, and available on internet, see
!     http://www.rug.nl/rc/hpcv/visualisation/komplot

! set precision parameters 
!===============================================================================
!------------------------------------------------------------------------------+
Module set_PRECISION
  !
  Integer, Parameter :: wp = Kind (1.0e0)! working prec., single
  ! Integer, Parameter :: wp = Kind (1.0d0) ! working prec., double
  Integer, Parameter :: sp = Kind (1.0e0)! single prec.
  Integer, Parameter :: dp = Kind (1.0d0)! double prec.
  !
End Module set_PRECISION
!------------------------------------------------------------------------------+
!===============================================================================

!
! set flag whether or not program runs in debug mode
!===============================================================================
!------------------------------------------------------------------------------+
Module DEBUG_FLAG
  Logical, Parameter :: DEBUG = .True.
  !       Logical, Parameter :: DEBUG = .False. 
End Module DEBUG_FLAG
!------------------------------------------------------------------------------+
!===============================================================================

! Module containing various plot subroutines for Chapter 7
!===============================================================================
!------------------------------------------------------------------------------+
Module Plots
  !
  ! Contains subroutines:
  ! conf1, halo1, dlim1, tae_3703, elm_3703, divheat1, magn1
  !
  ! These correspond to execises 1, 2, 3, 4, 5 and 7 of Chapter 7.
  ! (N.B.: the plots of exercise 6 have been made by a SAS program)
  !
  Implicit None

Contains

  ! Confinement time (Exercise 1)
  !=============================================================================
  Subroutine conf1
    ! Date: July 15, 2001
    ! Update: May 2004
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from conf1.dat and making plots for Chapter 7
    !
    !
    ! Imported procedures:
    !
    Use KOMPLOT_EXT !, Only: sec_axis
    !
    ! Imported parameters:
    !
    Use set_PRECISION
    Use DEBUG_FLAG

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    Integer, Parameter :: nmax=100, p=5
    Integer, Parameter :: n_desc_lines=5
    Integer            :: ios,i,n,fmt_h,fmt_v
    Real(sp)           :: xs,xl,xh,ys,yl,yh,sf_ma, sf_an, sf_an_sec
    ! meaning of some variables:
    ! n is upper bound of number of cases, p is the exact number of variables
    !sf stands for scale factor

    ! Arrays:
    ! correct path name for the input datafile has to be provided:
    Character (LEN=48), Parameter :: inp='../conf1.dat' 
    Character (LEN=40) :: description
    Character (LEN=24) :: ax_label
    Character (LEN=1)  :: ax_type

    ! single precision is used for real variables:
    Real(sp) :: x(1:nmax,1:p)
    Real(sp) :: wth(nmax),   ip(nmax),   nel(nmax),   bt(nmax),   pl(nmax)
    Real(sp) :: elwth(nmax), elip(nmax), elnel(nmax), elbt(nmax), elpl(nmax)

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    Do i=1, n_desc_lines
       Read (df,*,IOSTAT=ios) description
    End Do

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    If (debug) Then
       Write(out,*) n, 'observations read'
    End If

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    wth(1:n) = x(1:n,1)
    ip (1:n) = x(1:n,2)
    bt (1:n) = x(1:n,3)
    nel(1:n) = x(1:n,4)
    pl (1:n) = x(1:n,5)

    ! logarithms:
    elwth(1:n) = Log(wth(1:n));
    elip(1:n)  = Log(ip(1:n));
    elbt(1:n)  = Log(bt(1:n));
    elnel(1:n) = Log(nel(1:n));
    elpl(1:n)  = Log(pl(1:n));

    ! just to show something
    Write(out,*) 'program conf1, starting KOMPLOT'

    ! composing plots, general options: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,24.)
    !  CALL frame_style('scientific')
    Call suppress_border(1) !no border

    !  CALL gridlines(1,1)

    sf_ma=0.7
    sf_an=1.2
    sf_an_sec=1.15

    Call scale_markers(sf_ma)   
    Call scale_annotations(sf_an)
    Call x_margins_f90            (xm_left=2.0,xm_right=1.0)
    Call format_x_axisnumbers_f90 (n_dec_pos=1,n_exp_10=0)
    Call format_y_axisnumbers_f90 (1,0)

    xs=5.5 ; ys=9.0 ! size of graphs
    fmt_h=201       ! format of sec. hor.  axis numbers
    fmt_v=202       ! format of sec. vert. axis numbers 

    ! usage secondary axis:
    ! SUBROUTINE sec_axis(axis_type, xl, xh, yl, yh, lab, & 
    !                     ax_pos, tick_pct, num_fmt, num_par_pct, num_per_pct, &
    !                     lab_par_pct, lab_per_pct,sf_an, debug)


    ! plot 1:-------------------------------------------------------------------
    Call shift_frame_in_page(0.0,12.0)
    xl=-0.8 ; xh=0.4
    yl=-3.0 ; yh=-1.0
    Call tick_distances(xs/4,ys/4)
    Call frame(xs,xl,xh,ys,yl,yh, &
         &           'ln (Ip)','ln (Wth)',' ')   
    Call markers(' ',n, elip, elwth, 17)

    ! secondary axes:
    ax_type='h'
    ax_label='Ip [MA]'
    Call sec_axis(ax_type, xl, xh, yl, yh, ax_label,   &
         &                num_fmt=fmt_h, sf_an=sf_an_sec)
    Call sec_axis('v',     xl, xh, yl, yh, 'Wth [MJ]', &
         &                 num_fmt=fmt_v, sf_an=sf_an_sec)

    ! plot 2:------------------------------------------------------------------- 
    Call shift_frame_in_page(9.5,12.0)
    xl=0.4 ;  xh= 1.6
    yl=-3.0 ; yh=-1.0
    Call tick_distances(xs/4.0,ys/4.0)  
    Call frame(xs,xl,xh,ys,yl,yh, &
         &           'ln (Bt)','ln (Wth)',' ')   
    Call markers(' ',n, elbt, elwth, 17)  
    Call sec_axis('h',     xl, xh, yl, yh, 'Bt [T]',   &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v',     xl, xh, yl, yh, 'Wth [MJ]', &
         &               num_fmt=fmt_v, sf_an=sf_an_sec)

    ! plot 3:-------------------------------------------------------------------
    Call shift_frame_in_page(0.0,0.0)
    xl=0.8 ;  xh= 2.8
    yl=-3.0 ; yh=-1.0
    Call tick_distances(xs/4.0,ys/4.0)
    Call frame(xs,xl,xh,ys,yl,yh, &
         &           'ln (ne)','ln (Wth)',' ')   
    Call markers(' ',n, elnel, elwth, 17)  
    Call sec_axis('h',     xl, xh, yl, yh, 'ne [10^19/m^3]', &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v',     xl, xh, yl, yh, 'Wth [MJ]',       &
         &               num_fmt=fmt_v, sf_an=sf_an_sec)

    ! plot 4:-------------------------------------------------------------------
    Call shift_frame_in_page(9.5,0.0)
    xl=0.0 ;  xh= 2.4
    yl=-3.0 ; yh=-1.0
    Call tick_distances(xs/4.0,ys/4.0)
    Call frame(xs,xl,xh,ys,yl,yh, &
         &           'ln (PL)','ln (Wth)',' ')
    Call markers(' ',n, elpl, elwth, 17)  
    Call sec_axis('h',     xl, xh, yl, yh, 'PL [MW]',  &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v',     xl, xh, yl, yh, 'Wth [MJ]', &
         &               num_fmt=fmt_v, sf_an=sf_an_sec)

    ! producing postscript file:
    Call komplot(50,'conf1.eps')

  End Subroutine conf1
  !=============================================================================

  ! Halo current (Exercise 2)
  !=============================================================================
  Subroutine halo1
    !---------------------------------------------------------------------------
    ! Subroutine: halo1
    ! Date: July 15, 2001
    ! Update;
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from divheat1.dat and making plots for Chapter 7
    !

    ! Imported procedures:

    Use komplot_ext, Only: sec_axis
    Use DEBUG_FLAG

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    ! n is upper bound of number of cases, p is the exact number of variables:
    Integer, Parameter :: nmax=100, p=10
    Integer, Parameter :: n_desc_lines=4
    Integer            :: ios,i,n,solid_dot
    Real               :: xs,xl,xh,ys,yl,yh,sf_ma,sf_an,sf_an_sec
    Real               :: xp,yp,height,angle,glab_xpct,glab_ypct

    ! Arrays:
    Character (LEN=48), Parameter :: inp = '../halo1pre.dat' 
    Character (LEN=40) :: description
    Real :: x(1:nmax,1:p)
    Real :: shot(nmax),  maxI_o(nmax), maxI_i(nmax), Idt_o(nmax), Idt_i(nmax), &
    &       torasya(nmax), torasyi(nmax), polasy(nmax), ip(nmax), bt(nmax)
    Real :: elIdt_o(nmax), elIdt_i(nmax), elIdt(nmax), elip(nmax), elbt(nmax)
    Integer :: isyma(nmax)
    !REAL, ALLOCATABLE :: wth(:), ip(:), nel(:), bt(:), pl(:)
    !REAL, ALLOCATABLE :: elwth(:), elip(:), elnel(:), elbt(:), elpl(:)

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    Do i=1, n_desc_lines
       Read (df,*,IOSTAT=ios) description
    End Do

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    shot   (1:n) = x(1:n,1)
    maxI_o (1:n) = x(1:n,2)
    maxI_i (1:n) = x(1:n,3)
    Idt_o  (1:n) = x(1:n,4)
    Idt_i  (1:n) = x(1:n,5)
    torasya(1:n) = x(1:n,6)
    torasyi(1:n) = x(1:n,7)
    polasy (1:n) = x(1:n,8)
    ip     (1:n) = x(1:n,9)
    bt     (1:n) = x(1:n,10)

    ! logarithms:
    elip    (1:n)  = Log(Abs(ip(1:n))/10.0**6) !MA
    elbt    (1:n)  = Log(Abs(bt(1:n)))
    elIdt_o (1:n)  = Log(Idt_o(1:n)/10.0**3)  !MA/s (presumably)
    elIdt_i (1:n)  = Log(Idt_i(1:n)/10.0**3)
    elIdt   (1:n)  = Log((Idt_i(1:n)+Idt_o(1:n))/10.0**3)

    ! recalculate polasy to get the sign:
    polasy(1:n) = (Idt_i(1:n) - Idt_o(1:n)) / (Idt_i(1:n) + Idt_o(1:n))

    ! set marker symbols
    Do i=1,n
       If (polasy(i) .Lt. -0.15) Then 
          isyma(i) = 19 ! triangle down 
       Else If (polasy(i) .Gt. 0.15) Then
          isyma(i) = 21 ! triangle up
       Else 
          isyma(i)= 17  ! dot
       End If
    End Do

    ! just to show something
    Write(out,*) 'program halo1, starting KOMPLOT'
    If (debug) Then
       Write(out,*) 'x(1:3,9)', x(1:3,9)
       Write(out,*) 'ip(1:3)', ip(1:3)
       Write(out,*) 'elip(1:3)', elip(1:3)
    End If

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,24.)
    !  CALL frame_style('scientific')
    Call suppress_border(1)

    !  CALL gridlines(1,1)

    sf_ma=0.7
    sf_an=1.2
    sf_an_sec=1.15
    glab_xpct = 15
    glab_ypct = 8.5
    solid_dot = 17

    Call scale_markers(sf_ma)  
    Call scale_annotations(sf_an)
    Call x_margins(2.0,1.0) 

    ! size of graphs
    xs=5.5 ; ys=9.0 

    ! general parameters of alphanumeric label for plot:

    height = 0.28*sf_an         
    angle  = 0.0 

    ! plot 0, poloidal asymmetry, plotted in the lower right corner  
    Call shift_frame_in_page(10.0,0.0)
    xl=-2.5 ; xh=0.5
    yl=-2.5 ; yh=0.5
    Call tick_distances(xs/4,ys/4)
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln(Idt_o)','ln(Idt_i)',' ')   
    Call var_markers(' ', n, elIdt_o, elIdt_i, isyma)
    xp=xl-glab_xpct*sf_an*(xh-xl)/100 
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(c)')      
    Call sec_axis('h', xl, xh, yl, yh, 'Idt_o [MA/s]',                         &
         &               num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'Idt_i [MA/s]',                         &
         &               num_fmt=201, lab_per_pct=17.0, sf_an=sf_an_sec)

    ! plots 1 to 2 are response variable against regression variables
    ! plot 1
    Call shift_frame_in_page(0.0,12.0)
    xl=-1.2 ; xh=0.3
    yl=-2.0 ; yh=1.0
    Call tick_distances(xs/5,ys/3)
    Call format_x_axisnumbers(1,0)
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln(Ip)','ln(Idt)',' ')   
    Call markers(' ', n, elip, elIdt, solid_dot)
    xp=xl-glab_xpct*sf_an*(xh-xl)/100 
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0   
    Call gstrng (xp,yp,height,angle,'(a)')   
    Call sec_axis('h', xl, xh, yl, yh, 'Ip [MA]',                              & 
         &               num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'Idt [MA/s]',                           & 
         &               num_fmt=201, lab_per_pct=17.0, sf_an=sf_an_sec)

    ! plot 2  
    Call shift_frame_in_page(9.5,12.0)
    xl=-0.2 ; xh=0.8
    yl=-2.0 ; yh=1.0
    Call tick_distances(xs/5,ys/4)
    Call format_x_axisnumbers(1,0)
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln(Bt)','ln(Idt)',' ')   
    Call markers(' ', n, elbt, elIdt, solid_dot)
    xp=xl-glab_xpct*sf_an*(xh-xl)/100 
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0   
    Call gstrng (xp,yp,height,angle,'(a)') 
    Call sec_axis('h', xl, xh, yl, yh, 'Bt [T]',                               &
         &               num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'Idt [MA/s]',                           &
         &               num_fmt=201, lab_per_pct=17.0, sf_an=sf_an_sec)

    ! plot 3, regression variables:

    Call shift_frame_in_page(0.0,0.0)
    xl=-1.2 ; xh=0.3
    yl=-0.2 ; yh=0.8
    Call tick_distances(xs/5,ys/5)
    Call format_x_axisnumbers(1,0)
    Call format_y_axisnumbers(1,0)
    Call frame(xs,xl,xh,ys,yl,yh,                                              &      
         &           'ln(Ip)','ln(Bt)',' ')
    Call markers(' ', n, elip, elbt, solid_dot)
    xp=xl-glab_xpct*sf_an*(xh-xl)/100 
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0    
    Call gstrng (xp,yp,height,angle,'(b)')
    Call sec_axis('h', xl, xh, yl, yh, 'Ip [MA]',                              &
         &               num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'Bt [T]',                               &
         &               num_fmt=201, lab_per_pct=17.0, sf_an=sf_an_sec)

    ! producing postscript file:
    Call komplot(50,'halo1.eps')

  End Subroutine halo1
  !=============================================================================

  ! Density limit (Exercise 3)
  !=============================================================================
  Subroutine dlim1
    !---------------------------------------------------------------------------
    ! Subroutine: dlim1
    ! Date: July 15, 2001
    ! Update: Dec. 6, 2001 (dlim1.dat, including new shots 
    !                       and new variable shotbo)
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from dlim1.dat and making plots for Chapter 7
    !

    ! Imported procedures:

    Use set_Precision                 
    Use komplot_ext, Only: sec_axis   ! from komplot_ext_mod.f90 
    Use DEBUG_FLAG

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    ! n is upper bound of number of cases, p is the exact number of variables:
    Integer, Parameter :: nmax=100, p=12
    Integer, Parameter :: n_desc_lines=4
    Integer            :: ios,i,n, solid_dot, star, dot
    Real(sp)           :: xs,xl,xh,ys,yl,yh,sf_ma,sf_an,sf_an_sec
    Real(sp)           :: xp,yp,height,angle,glab_xpct,glab_ypct 

    ! Arrays:
    Character (LEN=48), Parameter :: inp='../dlim1.dat';
    Character (LEN=40) :: description
    Real(wp) :: x(1:nmax,1:p)
    Real(wp) :: shot(nmax),  gr(nmax), shotbo(nmax), n_since_bo(nmax) 
    Real(wp) :: ne_Max(nmax), tdl(nmax), n12(nmax), io(nmax)
    Real(wp) :: co(nmax),    ic(nmax),   cc(nmax),  dco(nmax), dcc(nmax)
    Real(sp) :: lne_Max(nmax), ldco(nmax), ldcc(nmax)
    Integer  :: isyma(nmax)

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    Do i=1, n_desc_lines
       Read (df,*,IOSTAT=ios) description
    End Do

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    shot(1:n)   = x(1:n,1)
    gr(1:n)     = x(1:n,2)
    shotbo(1:n) = x(1:n,3)
    ne_Max(1:n) = x(1:n,4)
    tdl (1:n)   = x(1:n,5)
    n12 (1:n)   = x(1:n,6)
    io  (1:n)   = x(1:n,7)
    co  (1:n)   = x(1:n,8)
    ic  (1:n)   = x(1:n,9)
    cc  (1:n)   = x(1:n,10)
    dco (1:n)   = x(1:n,11)
    dcc (1:n)   = x(1:n,12)
    ! logarithms:
    lne_Max(1:n) = Log(ne_Max(1:n));
    ldco(1:n)    = Log(dco(1:n));
    ldcc(1:n)    = Log(dcc(1:n));
    ! other transformations:
    n_since_bo(1:n) = shot(1:n) - shotbo(1:n); 

    Write(out,*) 'program dlim1, starting KOMPLOT'
    ! just to show something
    If (debug) Then
       Write(out,*) 'n=',n
       Write(out,*) 'x(1:3,2)', x(1:3,2)
       Write(out,*) 'co(1:3)', co(1:3)
       Write(out,*) 'cc(1:3)', cc(1:3)
    End If

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,24.)
    !  CALL frame_style('scientific')
    Call suppress_border(1) 

    !  CALL gridlines(1,1)

    sf_ma=0.7
    sf_an=1.2
    sf_an_sec=1.15
    glab_xpct = 15
    glab_ypct = 8.5
    solid_dot = 17
    star      = 12
    dot       = 16

    ! set variable markers:

    Do i=1,n
       If (gr(i)==1) Then
          isyma(i) = solid_dot
       Else If (gr(i)==2) Then
          isyma(i) = star
       Else
          isyma(i) = dot
       End If
    End Do

    Call scale_markers(sf_ma)  
    Call scale_annotations(sf_an)
    Call x_margins(2.0,1.0) 

    ! size of graphs
    xs=5.5 ; ys=9.0 

    ! general parameters of alphanumeric label for plot:

    height = 0.28*sf_an           
    angle  = 0.0 

    ! plot 1, regression variables ---------------------------------------------
    Call x_margins(2.0,1.5) ! left and right
    Call shift_frame_in_page(0.0,0.0)
    Call tick_distances(xs/5,ys/4)
    xl=-4.0 ; xh=1.0
    yl=-1.2 ; yh=0.8
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln[O]','ln[C]',' ')   
    Call var_markers(' ',n, ldco, ldcc, isyma)   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100.0  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(b)') 
    Call sec_axis('h', xl, xh, yl, yh, '[O] (%)',                             &
         &  num_dens=2,        num_fmt=202, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, '[C] (%)',                             &
         &               num_fmt=201, lab_per_pct=20.0, sf_an=sf_an_sec)
    Call x_margins(2.0,1.0) 

    ! plot 2, univariate regression --------------------------------------------  
    Call shift_frame_in_page(0.0,12.0)
    Call tick_distances(xs/4,ys/5)
    xl=-1.2 ; xh=0.8
    yl= 0.9 ; yh=2.4
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln[C]','ln(ne_max)',' ')   
    Call var_markers(' ',n, ldcc, lne_max, isyma)  
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0   
    Call gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, '[C] (%)',                              &
         &               num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'ne_max [10^19/m^3]',                   &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)

    ! plot 3, univariate regression --------------------------------------------

    Call shift_frame_in_page(9.5,12.0)
    Call tick_distances(xs/5,ys/5)
    xl=-4.0 ; xh=1.0
    yl= 0.9 ; yh=2.4
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln[O]','ln(ne_max)',' ')
    Call var_markers(' ',n, ldco, lne_max, isyma)  
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0   
    Call gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, '[O] (%)',                              &
         &               num_fmt=202, num_dens=2, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'ne_max [10^19/m^3]',                   &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)

    ! producing postscript file:
    Call komplot(50,'dlim1.eps')

  End Subroutine dlim1
  !=============================================================================

  ! High frequency MHD modes, ELMs (Exercise 4A)
  !=============================================================================
  Subroutine elm_3703
    !---------------------------------
    ! Date: Dec 15, 2001
    ! Update;
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from elm_3703.dat and 
    ! making plots for Ch.4 using KOMPLOT
    !

    Use set_precision 

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    ! n is upper bound of number of cases, p is the exact number of variables:
    Integer, Parameter :: nmax=1200, p=28
    Integer, Parameter :: n_desc_lines=4
    Integer            :: ios,i,n

    ! Arrays:
    Character (LEN=48), Parameter :: inp='../elm_3703.dat';
    Character (LEN=40) :: description
    Real(wp) :: x(1:nmax,1:p)
    Real(sp) ::  time(1:nmax), &
         C09_01  (1:nmax), & 
         C09_02  (1:nmax), &
         C09_03  (1:nmax), &
         C09_04  (1:nmax), &
         C09_05  (1:nmax), &
         C09_06  (1:nmax), &
         C09_07  (1:nmax), &
         C09_09  (1:nmax), &
         C09_10  (1:nmax), &
         C09_11  (1:nmax), &
         C09_12  (1:nmax), &
         C09_14  (1:nmax), &
         C09_15  (1:nmax), &
         C09_16  (1:nmax), &
         C09_17  (1:nmax), &
         C09_18  (1:nmax), &
         C09_20  (1:nmax), &
         C09_21  (1:nmax), &
         C09_22  (1:nmax), &
         C09_23  (1:nmax), &
         C09_24  (1:nmax), &
         C09_26  (1:nmax), &
         C09_27  (1:nmax), &
         C09_28  (1:nmax), &
         C09_29  (1:nmax), &
         C09_30  (1:nmax), &
         C09_31  (1:nmax), &
         C09_32  (1:nmax) ;

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    If (n_desc_lines .Ge. 1) Then
       Do i=1, n_desc_lines
          Read (df,*,IOSTAT=ios) description
       End Do
    End If

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    C09_01  (1:n) = x(1:n,1)
    C09_02  (1:n) = x(1:n,2)
    C09_03  (1:n) = x(1:n,3)
    C09_04  (1:n) = x(1:n,4)
    C09_05  (1:n) = x(1:n,5)
    C09_06  (1:n) = x(1:n,6)
    C09_07  (1:n) = x(1:n,7)
    C09_09  (1:n) = x(1:n,8)
    C09_10  (1:n) = x(1:n,9)
    C09_11  (1:n) = x(1:n,10)
    C09_12  (1:n) = x(1:n,11)
    C09_14  (1:n) = x(1:n,12)
    C09_15  (1:n) = x(1:n,13)
    C09_16  (1:n) = x(1:n,14)
    C09_17  (1:n) = x(1:n,15)
    C09_18  (1:n) = x(1:n,16)
    C09_20  (1:n) = x(1:n,17)
    C09_21  (1:n) = x(1:n,18)
    C09_22  (1:n) = x(1:n,19)
    C09_23  (1:n) = x(1:n,20)
    C09_24  (1:n) = x(1:n,21)
    C09_26  (1:n) = x(1:n,22)
    C09_27  (1:n) = x(1:n,23)
    C09_28  (1:n) = x(1:n,24)
    C09_29  (1:n) = x(1:n,25)
    C09_30  (1:n) = x(1:n,26)
    C09_31  (1:n) = x(1:n,27)
    C09_32  (1:n) = x(1:n,28)

    ! linear time array:

    Do i=1,n
       time(i)=(2*i/1000.0)  ! in msec 
    End Do

    ! just to show something
    Write(out,*) 'program tae_3703, starting KOMPLOT'
    Write(out,*) 'n=', n

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,11.)
    Call frame_style('scientific')

    !  CALL gridlines(1,1)

    Call scale_markers(0.7)   
    Call x_margins(2.0,1.0)

    ! plot, overlay of two time traces:  
    Call shift_frame_in_page(0.0,0.0)
    Call tick_distances(16.0/5,9.0/5)
    Call frame(16.0,0.0,2.0,9.,10.0,10.0, &
         &           'time (ms)','C09_01 and C09_20',' ')   
    Call curve(' ', 1000, time, C09_01)
    Call curve(' ', 1000, time, C09_20) 

    ! For some reason, a stack-flow error occurs in postscript: 
    !   CALL shift_frame_in_page(0.0,0.0)
    !   CALL axis_intersection(1000.,0.)
    !   CALL tick_distances(16.0/5,9.0/5)
    !   CALL suppress_border(1)
    !   CALL suppress_axes(1,0)
    !   CALL reverse_axis_annotation(1,0)  
    !   CALL frame(16.0,0.0,1000,9.,10.0,10.0, &
    !        &           ' ','C09_20',' ')   
    !   CALL line_style_nr(2)
    !   CALL curve(' ', 5, time, C09_20)  

    ! producing postscript file:
    Call komplot(50,'elm_3703.eps')

  End Subroutine elm_3703
  !=============================================================================


  ! High frequency MHD modes, TAEs (Exercise 4A) 
  !=============================================================================
  Subroutine tae_3703
    !---------------------------------
    ! Date: July 15, 2001
    ! Update:
    ! Author: OJK/IPP
    !
    ! Purpose: reading data from tae_3703.dat and making plots for Chapter 7
    !

    Use set_Precision

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9, scf=8
    ! n is upper bound of number of cases, p is the exact number of variables:
    Integer, Parameter :: nmax=1200, p=28
    !Integer :: nmax, p (dynamic arrays are needed).
    Integer, Parameter :: n_desc_lines=4
    Integer            :: ios,i,n

    ! Arrays:
    Character (LEN=48), Parameter :: inp='../tae_3703.dat';
    Character (LEN=40) :: description
    Real(wp) :: x(1:nmax,1:p)
    Real(sp) ::  time(1:nmax), &
         C09_01  (1:nmax), & 
         C09_02  (1:nmax), &
         C09_03  (1:nmax), &
         C09_04  (1:nmax), &
         C09_05  (1:nmax), &
         C09_06  (1:nmax), &
         C09_07  (1:nmax), &
         C09_09  (1:nmax), &
         C09_10  (1:nmax), &
         C09_11  (1:nmax), &
         C09_12  (1:nmax), &
         C09_14  (1:nmax), &
         C09_15  (1:nmax), &
         C09_16  (1:nmax), &
         C09_17  (1:nmax), &
         C09_18  (1:nmax), &
         C09_20  (1:nmax), &
         C09_21  (1:nmax), &
         C09_22  (1:nmax), &
         C09_23  (1:nmax), &
         C09_24  (1:nmax), &
         C09_26  (1:nmax), &
         C09_27  (1:nmax), &
         C09_28  (1:nmax), &
         C09_29  (1:nmax), &
         C09_30  (1:nmax), &
         C09_31  (1:nmax), &
         C09_32  (1:nmax) ;

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    If (n_desc_lines .Ge. 1) Then
       Do i=1, n_desc_lines
          Read (df,*,IOSTAT=ios) description
       End Do
    End If

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    C09_01  (1:n) = x(1:n,1)
    C09_02  (1:n) = x(1:n,2)
    C09_03  (1:n) = x(1:n,3)
    C09_04  (1:n) = x(1:n,4)
    C09_05  (1:n) = x(1:n,5)
    C09_06  (1:n) = x(1:n,6)
    C09_07  (1:n) = x(1:n,7)
    C09_09  (1:n) = x(1:n,8)
    C09_10  (1:n) = x(1:n,9)
    C09_11  (1:n) = x(1:n,10)
    C09_12  (1:n) = x(1:n,11)
    C09_14  (1:n) = x(1:n,12)
    C09_15  (1:n) = x(1:n,13)
    C09_16  (1:n) = x(1:n,14)
    C09_17  (1:n) = x(1:n,15)
    C09_18  (1:n) = x(1:n,16)
    C09_20  (1:n) = x(1:n,17)
    C09_21  (1:n) = x(1:n,18)
    C09_22  (1:n) = x(1:n,19)
    C09_23  (1:n) = x(1:n,20)
    C09_24  (1:n) = x(1:n,21)
    C09_26  (1:n) = x(1:n,22)
    C09_27  (1:n) = x(1:n,23)
    C09_28  (1:n) = x(1:n,24)
    C09_29  (1:n) = x(1:n,25)
    C09_30  (1:n) = x(1:n,26)
    C09_31  (1:n) = x(1:n,27)
    C09_32  (1:n) = x(1:n,28)

    ! linear time array:

    Do i=1,n
       time(i)=(2*i/1000.0)  ! in msec 
    End Do

    ! just to show something
    Write(out,*) 'program tae_3703, starting KOMPLOT'
    Write(out,*) 'n=', n

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,11.) ! was 24.
    Call frame_style('scientific')

    !  CALL gridlines(1,1)

    Call scale_markers(0.7)   
    Call x_margins(2.0,1.0)

    ! plot, overlay of two time traces:  
    Call shift_frame_in_page(0.0,0.0)
    Call tick_distances(16.0/5,9.0/5)
    Call frame(16.0,0.0,1000,9.,10.0,10.0, &
         &           'time (ms)','C09_01 and C09_20',' ')   
    Call curve(' ', 1000, time, C09_01)
    Call curve(' ', 1000, time, C09_20) 

    ! producing postscript file:
    Call komplot(50,'tae_3703.eps')

  End Subroutine tae_3703
  !=============================================================================

  ! Divertor heat load (Exercise 5)
  !=============================================================================
  Subroutine divheat1
    !---------------------------------
    ! Date: July 15, 2001
    ! Update: March 1, 2003
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from divheat1.dat and making plots for Chapter 7
    !

    ! Imported procedures: 

    Use komplot_ext, Only: sec_axis
    Use debug_flag
    Use set_precision

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    Integer, Parameter :: nmax=100, p=6
    Integer, Parameter :: n_desc_lines=4
    Integer            :: ios,i,n, solid_dot
    Real(sp)           :: xs,xl,xh,ys,yl,yh,sf_ma,sf_an,sf_an_sec
    Real(sp)           :: xp,yp,height,angle,glab_xpct,glab_ypct 

    ! Arrays:
    Character (LEN=48), Parameter :: inp='../divheat1.dat';
    Character (LEN=40) :: description
    Real :: x(1:nmax,1:p)
    Real :: shot(nmax),  ne(nmax), q95(nmax), lambda(nmax) 
    Real :: q_plate(nmax), p_plate(nmax)
    Real :: lnel(nmax), lq95(nmax), llambda(nmax), lqplate(nmax), lpplate(nmax)

  !-End of subroutine header ---------------------------------------------------

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    Do i=1, n_desc_lines
       Read (df,*,IOSTAT=ios) description
    End Do

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    !  ALLOCATE (wth(n),ip(n),bt(n),nel(n),pl(n))

    ! variable names:
    shot   (1:n) = x(1:n,1)
    ne     (1:n) = x(1:n,2)/10.0**19 ! engineering unit
    q95    (1:n) = x(1:n,3)
    lambda (1:n) = x(1:n,4)
    q_plate(1:n) = x(1:n,5)/10.0**6 ! MW
    p_plate(1:n) = x(1:n,6)/10.0**6 ! MW

    ! logarithms:
    lnel   (1:n)  = Log(ne(1:n))
    lq95   (1:n)  = Log(q95(1:n))
    llambda(1:n)  = Log(lambda(1:n))
    lqplate(1:n)  = Log(q_plate(1:n))
    lpplate(1:n)  = Log(p_plate(1:n))

    ! just to show something
    Write(out,*) 'program divheat1, starting KOMPLOT'

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.,24.)
    !  CALL frame_style('scientific')
    Call suppress_border(1) 

    !  CALL gridlines(1,1)

    sf_ma=0.7
    sf_an=1.2
    sf_an_sec=1.15
    glab_xpct = 15
    glab_ypct = 8.5
    solid_dot = 17

    Call scale_markers(sf_ma)  
    Call scale_annotations(sf_an)
    Call x_margins(2.0,1.0) 

    ! size of graphs
    xs=5.5 ; ys=9.0 

    ! general parameters of alphanumeric label for plot:

    height = 0.28*sf_an           
    angle  = 0.0 

    ! plot 0, response variables, plotted in the upper right corner  
    Call shift_frame_in_page(10.0,12.5)
    Call tick_distances(xs/5,ys/5)
    xl= 2.0 ; xh=4.5
    yl=-1.0 ; yh=1.5
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln (lambda)','ln (q_plate)',' ')   
    Call markers(' ', n, llambda, lqplate, solid_dot)
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, 'lambda',                               &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'q_plate',                              &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! plots 1 to 3 are regression variables, plotted in left lower region
    ! plot 1
    Call shift_frame_in_page(0.0,12.0)
    Call tick_distances(xs/3,ys/4)
    xl= 1.0 ; xh=2.5
    yl= 0.8 ; yh=1.6
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln (ne)','ln (q95)',' ')   
    Call markers(' ', n, lnel, lq95, solid_dot)   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(b)')
    Call sec_axis('h', xl, xh, yl, yh, 'ne [10^19/m^3]',                       &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'q95',                                  &
         &               num_fmt=202, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! plot 2  
    Call shift_frame_in_page(0.0,0.0)
    Call tick_distances(xs/3,ys/4)
    xl= 1.0 ; xh=2.5
    yl= -2.0; yh=2.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln (ne)','ln (P_plate)',' ')   
    Call markers(' ', n, lnel, lpplate, solid_dot)  
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(b)')
    Call sec_axis('h', xl, xh, yl, yh, 'ne [10^19/m^3]',                       &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'P_plate',                              &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! plot 3

    Call shift_frame_in_page(9.5,0.0)
    Call tick_distances(xs/4,ys/4)
    xl= 0.8 ; xh=1.6
    yl= -2.0; yh=2.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln (q95)','ln (P_plate)',' ')
    Call markers(' ', n, lq95, lpplate, solid_dot)  
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(b)')
    Call sec_axis('h', xl, xh, yl, yh, 'q95',                                  &
         &               num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'P_plate',                              &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! producing postscript file:
    Call komplot(50,'divheat1.eps')

  End Subroutine divheat1
  !=============================================================================

  ! Plasma parameter recovery from external magnetic measurements (Exercise 7)
  !=============================================================================
  Subroutine magn1
    !----------------------------------------------------------
    ! Date: Jan 30, 2002
    ! Update: May 2004
    ! Author: O. Kardaun
    !
    ! Purpose: reading data from magn1.dat and making plots for Chapter 7
    !

    ! Imported procedures:

    Use set_precision                  
    Use komplot_ext, Only: sec_axis   ! from komplot_ext_mod.f90
    Use debug_flag

    Implicit None

    ! Scalars:
    Integer, Parameter :: eof=-1
    Integer, Parameter :: out=6, df=9
    ! n is upper bound of number of cases, p is the exact number of variables:
    Integer, Parameter :: nmax=2000, p=29
    Integer, Parameter :: n_desc_lines=7
    Integer            :: ios,i,n, solid_dot, star, dot
    Real(sp)           :: xs,xl,xh,ys,yl,yh,sf_ma,sf_an,sf_an_sec
    Real(sp)           :: xp,yp,height,angle,glab_xpct,glab_ypct 

    ! Arrays:
    Character (LEN=48), Parameter :: inp='../magn1.dat';
    Character (LEN=40) :: description
    ! everything sp instead of wp
    Real(sp) :: x(1:nmax,1:p)
    Real(sp) :: rin(nmax),  rout(nmax), rzmin(nmax), zmin(nmax)
    Real(sp) :: rzmax(nmax), zmax(nmax), bp(nmax), li(nmax), li_2(nmax)
    Real(sp) :: pc1(nmax), pc2(nmax), pc3(nmax), pc4(nmax), pc5(nmax)
    Real(sp) :: kappa(nmax), pc(1:20,1:nmax)
    ! always sp:
    Real(sp) :: lbp(nmax), lli(nmax), lli_2(nmax), lkappa(nmax)
    Integer  :: isyma(nmax)

  !-End of subroutine header ---------------------------------------------------
    ! set debugging flag:

    !  debug=.True.

    ! Reading ascii data file:

    Open (df, FILE=inp, ACTION='read', IOSTAT=ios)
    If (ios /= 0) Then
       Write(out,*) 'Error in opening file :', inp
    End If

    Do i=1, n_desc_lines
       Read (df,*,IOSTAT=ios) description
    End Do

    i=0
    Do While (i < nmax .And. ios == 0)
       i=i+1
       Read(df,*,IOSTAT=ios) x(i,1:p)
    End Do

    If (I >= nmax .Or. ios /= eof) Then
       Write(out,*) 'Error in reading file :', inp
    End If
    n=i-1

    ! variable names:

    rin(1:n)       =  x(1:n,1)
    rout(1:n)      =  x(1:n,2) 
    rzmin(1:n)     =  x(1:n,3)   
    zmin(1:n)      =  x(1:n,4)
    rzmax(1:n)     =  x(1:n,5) 
    zmax(1:n)      =  x(1:n,6) 
    bp(1:n)        =  x(1:n,7) 
    li(1:n)        =  x(1:n,8)
    kappa(1:n)     =  x(1:n,9)  
    pc(1:20,1:n)   =  Transpose(x(1:n,10:29))

    ! transformations:
    li_2(1:n)  = li(1:n)/2.0
    pc1(1:n)   = pc(1,1:n)/1000.0 ! later perhaps 2-d arrays for KOMPLOT
    pc2(1:n)   = pc(2,1:n)/1000.0

    lbp(1:n)    = Log(bp(1:n)) 
    lli(1:n)    = Log(li(1:n))
    lli_2(1:n)  = Log(li_2(1:n))
    lkappa(1:n) = Log(kappa(1:n)) 

    Write(out,*) 'program magn1, starting KOMPLOT'
    ! just to show something
    If (debug) Then
       Write(out,*) 'n=',n
       Write(out,*) 'x(1:3,2)',   x(1:3,2)
       Write(out,*) 'li(1:3)',    li(1:3)
       Write(out,*) 'kappa(1:3)', kappa(1:3)
       Write(out,*) 'pc1(1:3)', pc1(1:3)
       Write(out,*) 'pc2(1:3)', pc2(1:3)
    End If

    ! composing plots: 

    Call scale_dimensions(1.0)
    Call page_dimensions(19.25,24.)
    !  Call frame_style('scientific')
    Call suppress_border(1) 

    sf_ma=0.2
    sf_an=1.2
    sf_an_sec=1.2
    glab_xpct = 15
    glab_ypct = 8.5
    solid_dot = 17

    Call scale_markers(sf_ma)  
    Call scale_annotations(sf_an)
    Call x_margins(1.8,1.5) ! left and right
    Call y_margins(1.5,1.5) ! bottom / top
    Call font_switch_character('$') 

    ! size of graphs
    xs=6.0 ; ys=5.0 

    ! general parameters of alphanumeric label for plot:

    height = 0.28*sf_an           
    angle  = 0.0 

    ! plot 1, upper left -------------------------------------------------------
    Call shift_frame_in_page(0.0,16.0)
    Call tick_distances(xs/5,ys/7)
    xl=-0.2 ; xh=0.8
    yl=-2.5 ; yh=1.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln ($Mk$A)','ln (li/2)',' ')   
    Call markers(' ',n, lkappa, lli_2, solid_dot)   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, '$Mk$A',                                &
         &               num_fmt=201, lab_per_pct = 15.0, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'li/2',                                 &
         &               num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! plot 2, middle left ------------------------------------------------------
    Call shift_frame_in_page(0.0,8.0)
    Call tick_distances(xs/6,ys/7)
    xl=-4.0 ; xh=2.0
    yl=-2.5 ; yh=1.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'ln ($Mb$Ap)','ln (li/2)',' ')   
    Call markers(' ',n, lbp, lli_2, solid_dot)   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100.0  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    !  CALL gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, '$Mb$Ap',                              &
         &        num_fmt=202, lab_per_pct = 15.0, num_dens=2, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'li/2',                                &
         &        num_fmt=201, lab_per_pct=18.0, sf_an=sf_an_sec)    


    ! plot 3,  lower left -----------------------------------------------------
    Call shift_frame_in_page(0.0,0.0)
    Call tick_distances(xs/6,ys/5)
    xl=-4.0 ; xh=2.0
    yl=-.25 ; yh=1.0
    Call frame(xs,xl,xh,ys,yl,yh,                                             &
         &           'ln ($Mb$Ap)','ln ($Mk$A)',' ')   
    Call markers(' ',n, lbp, lkappa, solid_dot)   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100.0  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    !  CALL gstrng (xp,yp,height,angle,'(a)')
    Call sec_axis('h', xl, xh, yl, yh, '$Mb$Ap',                              &
         &        num_fmt=201, lab_per_pct = 15.0, num_dens=2, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, '$Mk$A',                               &
         &        num_fmt=202, lab_per_pct=18.0, sf_an=sf_an_sec)    

    ! left hand part of A4-paper:

    Call frame_style('scientific')
    Call x_margins(0.50,1.5)  ! left and right
    Call y_margins(1.5,1.5)   ! bottom / top
    Call scale_markers(sf_ma) ! not changed from sf_ma=0.2
    Call scale_annotations(sf_an)
    Call font_switch_character('$') 

    ! plot 4,  upper right -----------------------------------------------------
    Call shift_frame_in_page(11.25,16.0)
    Call tick_distances(xs/6,ys/4)
    xl=-15.0 ; xh=15.0
    yl=-3.0  ; yh= 1.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'pc1','ln (li/2)',' ')   
    Call markers(' ',n, pc1, lli_2, solid_dot)
    glab_xpct = 6.5
    glab_ypct = 6.5   
    xp=xl-glab_xpct*sf_an*(xh-xl)/100.0  
    yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    Call gstrng (xp,yp,height,angle,'(b)')
    !  CALL sec_axis('h', xl, xh, yl, yh, 'beta(pol)',                         &
    !  &             num_fmt=201, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'li/2',                                 &
         &        num_fmt=202, lab_per_pct=18.0, num_dens=2, sf_an=sf_an_sec)  

    ! plot 5,  middle right ----------------------------------------------------
    Call shift_frame_in_page(11.25,8.0)
    Call tick_distances(xs/6,ys/4)
    xl=-15.0 ; xh=15.0
    yl=-3.0  ; yh=1.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'pc1','ln (li/2+$Mb$Ap)',' ')   
    Call markers(' ',n, pc1, lli_2, solid_dot)   
    !  xp=xl-glab_xpct*sf_an*(xh-xl)/100.0  
    !  yp=yh+glab_ypct*sf_an*(yh-yl)/100.0  
    !  CALL gstrng (xp,yp,height,angle,'(a)')
    !  CALL sec_axis('h', xl, xh, yl, yh, 'beta(pol)',                         &
    ! &              num_fmt=200, sf_an=sf_an_sec)
    Call sec_axis('v', xl, xh, yl, yh, 'li/2+$Mb$Ap',                          &
         &        num_fmt=202, lab_per_pct=18.0, num_dens=2, sf_an=sf_an_sec) 

    ! plot 6,  bottom right ----------------------------------------------------
    Call shift_frame_in_page(11.25,0.0)
    !  Call change(2,-1.0,0.0) ! shift origin xx=1cm to the left
    Call tick_distances(xs/6,ys/5)
    xl=-15.0 ;  xh=15.0
    yl=-15.0  ; yh=15.0
    Call frame(xs,xl,xh,ys,yl,yh,                                              &
         &           'pc1','pc2','')
    !  Call gstrng(-18.0,-4.0,1.0,0.0,'pc2')   
    Call markers(' ',n, pc1, pc2, solid_dot)   

    ! producing postscript file:
    Call komplot(50,'magn1.eps')

  End Subroutine magn1
  !=============================================================================

End Module Plots
!------------------------------------------------------------------------------+
!===============================================================================

! include komplot extension module source:
  Include 'komplot_ext_mod.f90'

! Main F-90/95 program
Program chap7
!===============================================================================
!===============================================================================
!
! Note: procedure calls for different plots below should be 
!  activated by uncommenting, and de-activated by commenting,
!  the relevant lines below
!
! Imported Procedures (uncomment if needed)
  Use KOMPLOT_EXT  ! from komplot_ext_mod.f90
!
  Use Plots, Only: conf1, halo1, dlim1, tae_3703, elm_3703, divheat1, magn1 
!
  Implicit None 
!
!- End of header (program chap7)------------------------------------------------
!
  Call conf1

  Call halo1

  Call dlim1

  Call tae_3703

  Call elm_3703

  Call divheat1

  Call magn1

End Program chap7
!===============================================================================
!===============================================================================
