#include "de.h"
#include "math.h"

//------global variable for the objective function--------------
float gfa_bound[21];

//------objective function---------------------------------------

t_pop evaluate(int i_D, t_pop t_tmp, long *l_nfeval, t_pop *tpa_array, int i_NP)
/**C*F****************************************************************
**                                                                  
** Function       :t_pop evaluate(int i_D, t_pop t_tmp, long *l_nfeval,
**                                t_pop *tpa_array, int i_NP)                                        
**                                                                  
** Author         :Rainer Storn                                     
**                                                                  
** Description    :Evaluates the actual cost function (objective function)
**                 In this case: min f(X) = exp(x1*x2*x3*x4*x5)
**                               subject to
**                               x1^2 + x2^2 + x3^2 + x4^2 + x5^2 = 10
**                               x2*x3 - 5*x4*x5 = 0
**                               x1^3 + x2^3 = -1
**                               -2.3 <= xi <= 2.3, i=1,2
**                               -3.2 <= xj <= 3.2  j=3,4,5
**
**                 Known global minimum: 0.0539498473
**                                       x1 = -1.717143
**                                       x2 = +1.595709
**                                       x3 = +1.827247
**                                       x4 = -0.7636413
**                                       x5 = -0.7636450                                                                
** Functions      :-                                                
**                                                                  
** Globals        :- 
**                                                                  
** Parameters     :i_D         (I)    number of parameters
**                 t_tmp       (I)    parameter vector
**                 l_nfeval   (I/O)   counter for function evaluations
**                 tpa_array   (I)    pointer to current population (not needed here)
**                 i_NP   	   (I)    number of population members (not needed here)   
**                                                                  
** Preconditions  :-                     
**                                                                  
** Postconditions :- 
**
** Return Value   :TRUE if trial vector wins, FALSE otherwise.                                            
**                                                                  
***C*F*E*************************************************************/
{
   int   i,j;
   float result, fa_x[5], fa_r[5];

   (*l_nfeval)++;

   for (i=0; i<5; i++)
   {
     fa_x[i] = t_tmp.fa_vector[i]; //just for ease of readability
   }

   //----cost function value to minimize-----------
   result = 1.0;
   for (i=0; i<i_D; i++)
   {
	   result *= fa_x[i];
   }

   t_tmp.fa_cost[0] = exp(result);

   //----constraint 0 (must be = 0)------------------
   fa_r[0] = 0.0;
   for (i=0; i<i_D; i++)
   {
	   fa_r[0] += fa_x[i]*fa_x[i];
   }
   fa_r[0] = fa_r[0] - 10.0;


   //----constraint 1 (must be = 0)-----------------
   fa_r[1] = fa_x[1]*fa_x[2] - 5.0*fa_x[3]*fa_x[4];


   //----constraint 2 (must be = 0)-----------------
   fa_r[2] =   fa_x[0]*fa_x[0]*fa_x[0]
	        + fa_x[1]*fa_x[1]*fa_x[1] + 1.0;

   t_tmp.fa_cost[0] += fa_r[0]*fa_r[0]+fa_r[1]*fa_r[1]+fa_r[2]*fa_r[2];


   //----inequality constraints (must be < 0)-------
   t_tmp.fa_constraint[0] = -t_tmp.fa_vector[0] - 2.3;
   t_tmp.fa_constraint[1] = -t_tmp.fa_vector[1] - 2.3;
   t_tmp.fa_constraint[2] = +t_tmp.fa_vector[0] - 2.3;
   t_tmp.fa_constraint[3] = +t_tmp.fa_vector[1] - 2.3;

   t_tmp.fa_constraint[4] = -t_tmp.fa_vector[2] - 3.2;
   t_tmp.fa_constraint[5] = -t_tmp.fa_vector[3] - 3.2;
   t_tmp.fa_constraint[6] = -t_tmp.fa_vector[4] - 3.2;
   t_tmp.fa_constraint[7] = +t_tmp.fa_vector[2] - 3.2;
   t_tmp.fa_constraint[8] = +t_tmp.fa_vector[3] - 3.2;
   t_tmp.fa_constraint[9] = +t_tmp.fa_vector[4] - 3.2;

   return(t_tmp);
}

int left_vector_wins(t_pop trial, t_pop target)
/**C*F****************************************************************
**                                                                  
** Function       :int left_vector_wins(t_pop t_trial, t_pop t_target)                                        
**                                                                  
** Author         :Rainer Storn                                     
**                                                                  
** Description    :Selection criterion of DE. Decides when the trial
**                 vector wins over the target vector.                 
**                                                                  
** Functions      :-                                                
**                                                                  
** Globals        :-                                                
**                                                                  
** Parameters     :t_trial    (I)   trial vector
**                 t_target   (I)   target vector   
**                                                                  
** Preconditions  :-                     
**                                                                  
** Postconditions :- 
**
** Return Value   :TRUE if trial vector wins, FALSE otherwise.                                            
**                                                                  
***C*F*E*************************************************************/
{
	int i;
	int ctrial_satisfy  = TRUE;
	int ctarget_satisfy = TRUE;
	int constraints_improve = TRUE;

    //----all constraints are satisfied ?----------
	for (i=0; i<10; i++)
	{
		if (trial.fa_constraint[i] > VTR) ctrial_satisfy = FALSE;
	}
	for (i=0; i<10; i++)
	{
		if (target.fa_constraint[i] > VTR) ctarget_satisfy = FALSE;
	}

	//----make the decision dependend on the constraints-------
	if ((ctrial_satisfy == TRUE) && (ctarget_satisfy == TRUE))
	{
	   if (trial.fa_cost[0] <= target.fa_cost[0]) return(TRUE);
	   else return(FALSE);
	}

	if ((ctrial_satisfy == TRUE) && (ctarget_satisfy == FALSE))
	{
	   return(TRUE); //constraints satisfaction is always best, trial wins
	}

	if ((ctrial_satisfy == FALSE) && (ctarget_satisfy == TRUE))
	{
	   return(FALSE); //constraints satisfaction is always best, target wins
	}

	if ((ctrial_satisfy == FALSE) && (ctarget_satisfy == FALSE))
	{
	   for (i=0; i<10; i++)
	   {
		  if (trial.fa_constraint[i] > target.fa_constraint[i]) constraints_improve = FALSE;
	   }
	   if (constraints_improve == TRUE) return(TRUE);//trial wins if all constraints are equal or better
	   else return(FALSE);
	}
	return(FALSE); //default value (only compiler needs it)

}


