#include "de.h"
#include <math.h>

//------global variable for the objective function--------------
float gfa_bound[21]; //not used here

//------test functions-------------------------------------------
/***************************************************************
                                                            
            Test functions, by Kenneth Price, 2004             
                                                            
 A collection of standard test functions for global
 optimization methods are listed below.
 In order to easily integrate them into the DeWin application
 please	apply the interface as described in the following via
 the example of the sphere function:

 t_pop evaluate(int i_D, t_pop t_tmp, ... )
 {
   int i;
   double x[MAXDIM];

   //---we need this to provide the appropriate interface to the functions---
   for (i=0; i<i_D; i++)
   {
	   x[i] = (double)t_tmp.fa_vector[i];
   }

   (*l_nfeval)++;  //increment function evaluation count

   t_tmp.fa_cost[0] = sphere(i_D, x);

   return(t_tmp);
 }

***************************************************************/


double sphere(int n, double x[])
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   type: uni-modal; separable
	   initial upper bound = 100, initial lower bound = -100
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(0,0,...,0)
	*/
	
	int j;
	double sum=0;

	for (j=0; j<n; j++)
	{
		
		sum+=x[j]*x[j];
	}

	return (sum);
}

double ellipsoid(int n, double x[])  // Principal axes of ellipsoid grow in length as powers of 2.
{

	/*  valid for any dimension, n>0
		constraints: unconstrained
		type: uni-modal; separable
		initial upper bound = 100, initial lower bound = -100
		value-to-reach = f(x*)+1.0e-6
		f(x*) = 0.0, x*=(0,0,...,0)
	*/

	int i;
	double a=1.,sum=0;

	for (i=0; i<n; i++)
	{
		sum+=a*x[i]*x[i];
		a+=a;
	}

	return(sum);
}

double ridge(int n, double x[])
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   type: uni-modal; non-separable
	   initial upper bound = 100, initial lower bound = -100
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(0,0,...,0)
	*/

	double a=0,sum=0;
	int j;

	for (j=0; j<n; j++)
	{
		a+=x[j];
		sum+=a*a;
	}
	
	return (sum);
}

double Rosenbrock(int n, double x[])  // the generalized Rosenbrock function - a 2nd ICEO function
{
	/* valid for any dimension, n>1
	   constraints: unconstrained
	   type: uni-modal; non-separable
	   initial upper bound = 30, initial lower bound = -30
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(1,1,...,1)
	*/
	
	int j;
	double a,b,sum=0;

    for (j=0; j<n-1; j++)
	{
        a=x[j]*x[j]-x[j+1];
        b=1.-x[j];
        sum+=100.*a*a+b*b;
	}

	return (sum);
}

double Neumaier(int n, double x[])
{
	/* valid for any dimension, n>1
	   constraints: unconstrained
	   function type: multi-modal(?), one global optimum(?); non-separable
	   initial upper bound = n*n, initial lower bound = -n*n
	   value-to-reach = f(x*)+.5
	   f(x*)= -n(n+4)(n-1)/6; x*[j]=(j+1)(n-j), j=0,1,...,n-1
	*/

	int j=0;
	double sum=0;

	sum=(x[j]-1.)*(x[j]-1.);

	for (j=1; j<n; j++)
	{
		sum=sum+(x[j]-1)*(x[j]-1)-x[j]*x[j-1];
	}

	return (sum);
}

double Ackley(int n, double x[])
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   function type: multi-modal, one global optimum
	   initial upper bound = 30, initial lower bound = -30
	   value-to-reach = f(x*)+1.0e-6
	   f(x*)= 0.0; x*=(0,0,...,0)
	*/

	int j;
	double sum,sum1=0,sum2=0,pi=3.141592654;

	for (j=0; j<n; j++)
	{
		sum1+=x[j]*x[j];
		sum2+=cos(2.*pi*x[j]);
	}

	sum1/=(double)n;
	sum2/=(double)n;
	sum=-20.*exp(-.2*sqrt(sum1))-exp(sum2)+20.+2.718281828; // Janka (incorrectly?) uses -.02 not -.2

	return (sum);
}

double Griewangk(int n, double x[])
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   function type: multi-modal, one global optimum; non-separable
	   initial upper bound = 600, initial lower bound = -600
	   value-to-reach = f(x*)+1.0e-6
	   f(x*)= 0.0; x*=(0,0,...,0)
	*/
	
	int j;
	double sum=0,prod=1.;

	for (j=0; j<n; j++)
	{
		sum+=x[j]*x[j];
		prod*=cos(x[j]/sqrt((double)(j+1)));
	}

	sum=sum/4000.-prod+1.;

	return (sum);
}

double Rastrigin(int n, double x[])
{
	/* valid for any dimension, n>0
	   constraints:  unconstrained
	   type: multi-modal with one global minimum; separable
	   initial upper bound = 5.12, initial lower bound = -5.12
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(0,0,...,0)
	*/
	
	int j;
	double sum=0,pi=3.141592654;

	for (j=0; j<n; j++)
	{
		sum+=x[j]*x[j]-10.*cos(2.*pi*x[j])+10.;
	}

	return (sum);
}

double Schwefel(int n, double x[])  // normalized to be independent of dimension
{
	/* valid for any dimension, n>0
	   constraints: bound constrained
	   type: multi-modal with one global minimum; separable
	   initial upper bound = 500, initial lower bound = -500
	   value-to-reach = f(x*)+.01
	   f(x*) = -418.983; x*=(s,s,...,s), s=420.968746
	*/
	
	int j;
	double sum=0;

	for (j=0; j<n; j++)
	{
		sum-=x[j]*sin(sqrt(fabs(x[j])));
	}

	sum=sum/(double)n;

	return (sum);
}

double Tchebychev(int n, double x[])  // Storn's Tchebychev - a 2nd ICEO function - generalized version
{
	/* Valid for any n>2
	   constraints: unconstrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 2^n, initial lower bound = -2^n
	   value-to-reach = f(x*)+1.0e-8
	   f(x*)=0.0; x*=(128,0,-256,0,160,0,-32,0,1) (n=9)
	   x*=(32768,0,-131072,0,212992,0,-180224,0,84480,0,-21504,0,2688,0,-128,0,1) (n=17)
	*/

	int i,j;
	static int sample,do_once=1;
	double a=1.,b=1.2,px,y=-1,sum=0;
	static double dx,dy;

	if (do_once)
	{
		for (j=0; j<n-2; j++)
		{
			dx=2.4*b-a;
			a=b;
			b=dx;
		}

		sample=32*n;
		dy=2./(double)sample;
		do_once=0;
	}

	for (i=0; i<=sample; i++)
	{
		px=x[0];
		for (j=1; j<n; j++)
		{
			px=y*px+x[j];
		}
		if (px<-1 || px>1) sum+=(1.-fabs(px))*(1.-fabs(px));
		y+=dy;
	}

	for (i=-1; i<=1; i+=2)
	{
		px=x[0];
		for (j=1; j<n; j++)
		{
			px=1.2*px+x[j];
		}
		
		if (px<dx) sum+=px*px;
	}

	return sum;
}

double Epi_Michalewicz(int n, double x[])  // the epistatic Michalewicz function - a 2nd ICEO function
{ 
	/* valid for any dimension, n>1
	   type: multi-modal with one global minimum; non-separable
       constraints: bound constrained
	   initial upper bound = 3.1416, initial lower bound = 0
	   value-to-reach = f(x*)+.001	                    
	   f(x*) = -4.68766; x*=(2.693,.259,2.074,1.023,1.720) (d=5)
	   f(x*) = -9.66009; x*=(2.693,.259,2.074,1.023,2.275,.500,2.138,.794,2.819,1.571) (d=10)
	*/
		
	int j;
	double cost,sint,sum,y,m=10.,pi=3.141592654;

	cost = cos(pi/6.);
	sint = sin(pi/6.);

	sum = -sin(x[n-1])*pow(sin(n*x[n-1]*x[n-1]/pi),2.0*m); 

	for (j=0; j<n-1; j++)
	{
		if (j%2==1) y = x[j-1]*sint + x[j]*cost;
		else y = x[j]*cost - x[j+1]*sint;
		sum -= sin(y)*pow(sin((j+1)*y*y/pi),2.0*m); 

	}

	return(sum); 
}

double Langerman(int n, double x[])  // The modified Langerman function - a 2nd ICEO function
{
	/* valid for any dimension, n<11
	   constraints: unconstrained
	   type: multi-modal with one global optimum
	   initial upper bound = 10, initial lower bound = 0
	   value-to-reach = f(x*)+.001
	   f(x*) = -.965000; x*=(8.074,8.777,3.467,1.863,6.708) (n=5)
	                     x*=(8.074,8.777,3.467,1.863,6.708,6.349,4.534,.276,7.633,1.567) (n=10)
	*/

	int i,j;
	double dx,sum=0,dist,pi=3.141592654;
	double a[5][10] = {
		{9.681, 0.667, 4.783, 9.095, 3.517, 9.325, 6.544, 0.211, 5.122, 2.020},
		{9.400, 2.041, 3.788, 7.931, 2.882, 2.672, 3.568, 1.284, 7.033, 7.374},
		{8.025, 9.152, 5.114, 7.621, 4.564, 4.711, 2.996, 6.126, 0.734, 4.982},
		{2.196, 0.415, 5.649, 6.979, 9.510, 9.166, 6.304, 6.054, 9.377, 1.426},
		{8.074, 8.777, 3.467, 1.863, 6.708, 6.349, 4.534, 0.276, 7.633, 1.567}};
		
	double c[5] = {0.806,0.517,.1,0.908,0.965};

	for (i=0; i<5; i++)
	{
		dist=0;
		for (j=0; j<n; j++)
		{
			dx=x[j]-a[i][j];
			dist+=dx*dx;
		}
		sum-=c[i]*(exp(-dist/pi)*cos(pi*dist));
	}
	return (sum);
}

double Shekel(int n, double x[])  // The modified Shekel's Foxholes function - a 2nd ICEO function
{
	/* valid for any dimension, n<11
	   constraints: unconstrained
	   type: multi-modal with one global optimum; separable
	   initial upper bound = 10, initial lower bound = 0
	   value-to-reach = f(x*)+.01
	   f(x*) = -10.4056; x*=(8.0248,9.1519,5.1140,7.6209,4.5642) (n=5)
	   f(x*) = -10.2088; x*=(8.0248,9.1519,5.1140,7.6209,4.5642,4.7110,2.9960,6.1260,.7340,4.9820) (n=10)
	*/

	int i,j;
	double sum=0,sp,h;
	static double a[30][10] = {
		{9.681, 0.667, 4.783, 9.095, 3.517, 9.325, 6.544, 0.211, 5.122, 2.020},
		{9.400, 2.041, 3.788, 7.931, 2.882, 2.672, 3.568, 1.284, 7.033, 7.374},
		{8.025, 9.152, 5.114, 7.621, 4.564, 4.711, 2.996, 6.126, 0.734, 4.982},
		{2.196, 0.415, 5.649, 6.979, 9.510, 9.166, 6.304, 6.054, 9.377, 1.426},
		{8.074, 8.777, 3.467, 1.863, 6.708, 6.349, 4.534, 0.276, 7.633, 1.567},
		{7.650, 5.658, 0.720, 2.764, 3.278, 5.283, 7.474, 6.274, 1.409, 8.208},
		{1.256, 3.605, 8.623, 6.905, 4.584, 8.133, 6.071, 6.888, 4.187, 5.448},
		{8.314, 2.261, 4.224, 1.781, 4.124, 0.932, 8.129, 8.658, 1.208, 5.762},
		{0.226, 8.858, 1.420, 0.945, 1.622, 4.698, 6.228, 9.096, 0.972, 7.637},
		{7.305, 2.228, 1.242, 5.928, 9.133, 1.826, 4.060, 5.204, 8.713, 8.247},
		{0.652, 7.027, 0.508, 4.876, 8.807, 4.632, 5.808, 6.937, 3.291, 7.016},
		{2.699, 3.516, 5.874, 4.119, 4.461, 7.496, 8.817, 0.690, 6.593, 9.789},
		{8.327, 3.897, 2.017, 9.570, 9.825, 1.150, 1.395, 3.885, 6.354, 0.109},
		{2.132, 7.006, 7.136, 2.641, 1.882, 5.943, 7.273, 7.691, 2.880, 0.564},
		{4.707, 5.579, 4.080, 0.581, 9.698, 8.542, 8.077, 8.515, 9.231, 4.670},
		{8.304, 7.559, 8.567, 0.322, 7.128, 8.392, 1.472, 8.524, 2.277, 7.826},
		{8.632, 4.409, 4.832, 5.768, 7.050, 6.715, 1.711, 4.323, 4.405, 4.591},
		{4.887, 9.112, 0.170, 8.967, 9.693, 9.867, 7.508, 7.770, 8.382, 6.740},
		{2.440, 6.686, 4.299, 1.007, 7.008, 1.427, 9.398, 8.480, 9.950, 1.675},
		{6.306, 8.583, 6.084, 1.138, 4.350, 3.134, 7.853, 6.061, 7.457, 2.258},
		{0.652, 2.343, 1.370, 0.821, 1.310, 1.063, 0.689, 8.819, 8.833, 9.070},
		{5.558, 1.272, 5.756, 9.857, 2.279, 2.764, 1.284, 1.677, 1.244, 1.234},
		{3.352, 7.549, 9.817, 9.437, 8.687, 4.167, 2.570, 6.540, 0.228, 0.027},
		{8.798, 0.880, 2.370, 0.168, 1.701, 3.680, 1.231, 2.390, 2.499, 0.064},
		{1.460, 8.057, 1.336, 7.217, 7.914, 3.615, 9.981, 9.198, 5.292, 1.224},
		{0.432, 8.645, 8.774, 0.249, 8.081, 7.461, 4.416, 0.652, 4.002, 4.644},
		{0.679, 2.800, 5.523, 3.049, 2.968, 7.225, 6.730, 4.199, 9.614, 9.229},
		{4.263, 1.074, 7.286, 5.599, 8.291, 5.200, 9.214, 8.272, 4.398, 4.506},
		{9.496, 4.830, 3.150, 8.270, 5.079, 1.231, 5.731, 9.494, 1.883, 9.732},
		{4.138, 2.562, 2.532, 9.661, 5.611, 5.500, 6.886, 2.341, 9.699, 6.500}};
		static double c[30] = {
			0.806,
				0.517,
				0.1,
				0.908,
				0.965,
				0.669,
				0.524,
				0.902,
				0.531,
				0.876,
				0.462,
				0.491,
				0.463,
				0.714,
				0.352,
				0.869,
				0.813,
				0.811,
				0.828,
				0.964,
				0.789,
				0.360,
				0.369,
				0.992,
				0.332,
				0.817,
				0.632,
				0.883,
				0.608,
				0.326};

			for (i=0; i<30; i++)
			{
				sp=0;
				for (j=0; j<n; j++)
				{
					h=x[j]-a[i][j];
					sp+=h*h;
				}
				sum-=1.0/(sp+c[i]);
			}
			return (sum);
}

double odd_square(int n, double x[])  // also called, "vicious square" - a 2nd ICEO function
{
	/* valid for any dimension, n<21
	   type: multi-modal with many global minima; non-separable
       constraints: unconstrained
	   initial upper bound = 15, initial lower bound = -15
	   value-to-reach = f(x*)+.01	                    
	   f(x*) = -1.14383; x*=many solutions close to center point
	*/


	int j;
	double result,dist=0,dx=0,max_dist=0,pi=3.1415927;
	static double center_point[]={1., 1.3, .8, -.4, -1.3, 1.6, -.2, -.6, .5, 1.4,
		1., 1.3, .8, -.4, -1.3, 1.6, -.2, -.6, .5, 1.4};

	for (j=0; j<n; j++)
	{
		
		dx=fabs(x[j]-center_point[j]);
		dist+=dx*dx;
		if (dx>max_dist) max_dist=dx;
	}

	dist=sqrt(dist);
	max_dist=sqrt((double)n*max_dist*max_dist);

	result = -(exp(-max_dist/(2.*pi))*cos(pi*max_dist))*(1.+0.2*dist/(max_dist+.01));

	return (result);
}

double Salomon(int n, double x[])  // like ripples on a pond; rotationally invariant
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 100, initial lower bound = -100
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(0,0,...,0)
	*/
	
	int j;
	double a,result,sum=0,pi=3.141592654;
	
	for (j=0; j<n; j++)
	{
		sum+=x[j]*x[j];
	}

	a=sqrt(sum);
	result=-cos(2.*pi*a)+.1*a+1.;
	
	return (result);
}

double Lennard_Jones(int n, double x[])  // find the atomic configuration with minimum energy
{
	/* valid for any dimension, n=3*k, k=2,3,4,...,20.   k is the number of atoms in 3-D space
	   constraints: unconstrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 4, initial lower bound = -4
	   value-to-reach = minima[k-2]+.0001
	   f(x*) = minima[k-2]; see array of minima below; additional minima available at the
	   Cambridge cluster database: http://www-wales.ch.cam.ac.uk/~jon/structures/LJ/tables.150.html
	*/

	int i,j,k,a,b;
	double xd,yd,zd,ed,ud,sum=0;
	static double minima[]={-1.,-3.,-6.,-9.103852,-12.712062,-16.505384,-19.821489,-24.113360,
		-28.422532,-32.765970,-37.967600,-44.326801,-47.845157,-52.322627,-56.815742,-61.317995,
		-66.530949,-72.659782,-77.1777043};
	
	k=n/3;
	
	for (i=0; i<k-1; i++)
	{
		for (j=i+1; j<k; j++)
		{
			a=3*i;
			b=3*j;
			xd=x[a]-x[b];
			yd=x[a+1]-x[b+1];
			zd=x[a+2]-x[b+2];
			ed=xd*xd+yd*yd+zd*zd;
			ud=ed*ed*ed;
			if (ed>0) sum+=(1.0/ud-2.0)/ud;
		}
	}
	return (sum);
}

double Hilbert(int n, double x[])  // find the inverse of the (ill-conditioned) Hilbert matrix
{
	/* valid for any dimension, n=k*k, k=2,3,4,...
	   constraints: unconstrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 2^n, initial lower bound = -2^n
	   value-to-reach = f(x*)+1.0e-8
	   f(x*) = 0.0; x*={{9,-36,30},{-36,192,-180},{30,-180,180}} (n=9)
	   x*={{16,-120,240,-140},{-120,1200,-2700,1680},{240,-2700,6480,4200},{-140,1680,-4200,2800}} (n=16)
	*/
	
	int i,j,k,b;
	static int do_once=1;

	double sum=0;

	static double hilbert[10][10],y[10][10];			// increase matrix size if n>100

	b=(int)sqrt((double)n);

	if (do_once)
	{
		for (i=0; i<b; i++)
		{
			for (j=0; j<b; j++)
			{
				hilbert[i][j]=1./(double)(i+j+1);		// create a static Hilbert matrix
			}
		}
		
		do_once=0;
	}

	for (j=0; j<b; j++)
	{
		for (k=0; k<b; k++)
		{
			y[j][k]=0;
			for (i=0; i<b; i++)
			{
				y[j][k]+=hilbert[j][i]*x[k+b*i];		// compute matrix product H*x
			}
		}
	}
	

	for (i=0; i<b; i++)
	{
		for (j=0; j<b; j++)
		{
			if (i==j) sum+=fabs(y[i][j]-1);				// sum absolute value of deviations
			else sum+=fabs(y[i][j]);
		}
	}

	return (sum);
}

double Whitley(int n, double x[])  // in honor of D. Whitley; a composition of Rosenbrock and 1-D Griewangk
{
	/* valid for any dimension, n>0
	   constraints: unconstrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 100, initial lower bound = -100
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 0.0; x*=(1,1,...1)
	*/

	int i,j;
	double sum=0,a,b,v;
	
	for (i=0; i<n; i++)					// full matrix expansion that includes diagonal terms
	{
		for (j=0; j<n; j++)
		{
			
			a=x[i]*x[i]-x[j];
			b=1.-x[j];
			v=100.*a*a+b*b;
			sum=sum+v*v/4000.-cos(v)+1.; 
		}
	}
	
	return sum;
}

double Rana(int n, double x[])  // Whitley's E-F102 expands Rana's 2-D function with a minor diagonal wrap
{
	/* valid for any dimension, n>1
	   constraints: bound constrained
	   type: multi-modal with one global minimum; non-separable
	   initial upper bound = 511, initial lower bound = -512
	   value-to-reach = f(x*)+.01
	   f(x*) = -511.708; x*=(-512,-512,...,-512)
	*/

	int i,j;
	double sum=0;

	for (i=0; i<n; i++)
	{
		j=(i+1)%n;
		sum+=x[i]*sin(sqrt(fabs(x[j]+1.-x[i])))*cos(sqrt(fabs(x[j]+1.+x[i])))+
			(x[j]+1.)*cos(sqrt(fabs(x[j]+1.-x[i])))*sin(sqrt(fabs(x[j]+1.+x[i])));
	}

	return (sum/(double)n);
}

double Katsuura(int n, double x[])
{

	/* Valid for any dimension, n>0
	   constraints: unconstrained
	   type: multi-modal with one global minimum; separable
	   initial upper bound = 1000, initial lower bound = -1000
	   value-to-reach = f(x*)+1.0e-6
	   f(x*) = 1.0; x*=(0,0,...,0)
	*/

	int i,j;
	double sum,a,b,nint_b,prod=1;  // nint_b is the nearest integer to b

	//Note that at the first pass through this
	//function the parameters must be initialized
	//to [1,1000,-1000,1.0].

	for (i=0; i<n; i++)
	{
		sum=0;
		a=1.;
		for (j=0; j<32; j++)
		{
			b=a*x[i];
			nint_b=(int)b;
			if (b-nint_b > .5) nint_b++;
			sum+=fabs(b-nint_b)/a;
			a=a+a;
		}
		prod=prod*(1.+(double)(i+1)*sum);
	}
	
	return (prod);
}
//------objective function---------------------------------------

t_pop evaluate(int i_D, t_pop t_tmp, long *l_nfeval, t_pop *tpa_array, int i_NP)
/**C*F****************************************************************
**                                                                  
** Function       :t_pop evaluate(int i_D, t_pop t_tmp, long *l_nfeval,
**                                t_pop *tpa_array, int i_NP)                                        
**                                                                  
** Author         :Rainer Storn                                     
**                                                                  
** Description    :Evaluates the actual cost function (objective function).
**                 Functions from functions.h are used.                 
**                                                                  
** Functions      :-                                                
**                                                                  
** Globals        :- 
**                                                                  
** Parameters     :i_D         (I)    number of parameters
**                 t_tmp       (I)    parameter vector
**                 l_nfeval   (I/O)   counter for function evaluations
**                 tpa_array   (I)    pointer to current population (not needed here)
**                 i_NP   	   (I)    number of population members (not needed here)
**                                                                  
** Preconditions  :-                     
**                                                                  
** Postconditions :- 
**
** Return Value   :TRUE if trial vector wins, FALSE otherwise.                                            
**                                                                  
***C*F*E*************************************************************/
{
   int i;
   double x[MAXDIM];

   //---we need this to provide the appropriate interface to the functions---
   for (i=0; i<i_D; i++)
   {
	   x[i] = (double)t_tmp.fa_vector[i];
   }

   (*l_nfeval)++;  //increment function evaluation count

   t_tmp.fa_cost[0] = Ackley(i_D, x);

   return(t_tmp);
}

int left_vector_wins(t_pop t_trial, t_pop t_target)
/**C*F****************************************************************
**                                                                  
** Function       :int left_vector_wins(t_pop t_trial, t_pop t_target)                                        
**                                                                  
** Author         :Rainer Storn                                     
**                                                                  
** Description    :Selection criterion of DE. Decides when the trial
**                 vector wins over the target vector.                 
**                                                                  
** Functions      :-                                                
**                                                                  
** Globals        :-                                                
**                                                                  
** Parameters     :t_trial    (I)   trial vector
**                 t_target   (I)   target vector   
**                                                                  
** Preconditions  :-                     
**                                                                  
** Postconditions :- 
**
** Return Value   :TRUE if trial vector wins, FALSE otherwise.                                            
**                                                                  
***C*F*E*************************************************************/
{
	//---trial wins against target even when cost is equal.-----
	if (t_trial.fa_cost[0] <= t_target.fa_cost[0]) return(TRUE);
	else return(FALSE);
}


