function [L,N,l,R]=slcoil(D,d,b,x)
% SLCOIL     Single-Layer cylindrical COIL inductance calculation.
%            The original Lorentz 1879 formula, involving elliptic
%            integrals with positive and negative terms, is not vrey
%            practical due to the need of choosing the praticular
%            correction to the idealized long coil solution for each
%            form-factor case. Nagaoka (1906) numerical approximation,
%            using an alternating power series as a function of the 
%            form-factor R = D/b = 2a/b (D is the coil diameter, a is
%            the radius and b is the coil axial length), is restricted
%            to form-factors > 0.5. Other authors ( Rayleigh & Niven,
%            1881, Heaviside, 1889, Rosa, 1906, Butterworth, 1916,
%            Grover, 1923 & 1931) have given better algorithms for
%            other particular ranges and design optimizations (Ref.:
%            F.W.Grover, Inductance calculation, Van Nostrand, 1946,
%            reprinted by Instrument Society of America, 1973).
%            The Margan (1987) approximation is extremely simple and
%            valid (error < 1%) within form-factor range 0.01 to 100.
%
%            function call :     [L,N,l,R]=slcoil(D,d,b,x)
%
%            where the output parameters are :
%                      L = coil inductance in H (Henry)
%                      N = total number of turns
%                      l = total length of wire in m
%                      R = D/b = form-factor
%            and the input parameters are :
%                      D = 2a = coil diameter in m
%                      d = wire diameter in m
%                      b = coil axial length in m
%                      x = distance between adjacent turns
%                          ( 0 < x < d ). Optional.
%
%            (Note : If D, d, b are entered in mm, then L is in mH
%                    and the total wire length is also in mm.)

% by Erik Margan, 08.07.1987.

if nargin < 4
    % the function was called with no x
    % ==> assume a tightly wound coil ( x << d )
    x = 0 ;
end

if nargin < 3
    % if b is not given, use
    % R = 2.451 : the most ecconomical form,
    % maximum inductance for a given wire length
    b = D / 2.451 ;
end

% free-space magnetic permeability [Vs/Am]
u = 4 * pi * 1e-7 ;

% form-factor
R = D / b ;

% correction for "short" coils ( D/b > 0.01 )
k = 2 / ( 2 + R^( sin( pi / 3 ) ) ) ;

% inductance in mH (miliHenry) ( error < 1% for 0.01 < R < 100 )
L = k * u * pi * D^2 * b / ( 4 * ( d + x )^2 ) ;

% total number of turns
N = b / ( d + x ) ;

% total wire length in mm
l = pi * D * N ;
