function [z,p,k]=bestap(N,x)
%BESTAP  BESsel-Thomson Analog Poles or Polynomial coefficients.
%	 Use :
%	 [z,p,k]=bestap(N)	or
%	 [z,p,k]=bestap(N,'n')	returns the zeros, poles and gain of the N-th
%				order system, normalized to have the -3dB upper
%				frequency response limit at 1Hz. This is an all
%				-pole system, so an empty matrix is returned in
%				z. The poles are calculated after the condition
%				for a maximaly flat envelope (group) delay.
%	 
%	 [z,p,k]=bestap(N,'t')	normalizes the poles for the equal time delay
%				(Td=1 or "bandwidth improovement") condition.
%
%	 [z,p,k]=bestap(N,'a')	normalizes the poles to have the same frequency
%				response asymptote as the Butterworth system of
%				equal order.
%
%	 c=bestap(N)		returns the N-th order polynomial coefficients.
%
%	 See SYSVIEW to plot resulting system functions.
%	 See also COMPFILT and FAMILY.

%    Erik Margan, Ljubljana, Slovenia
%	 October 12th, 1988

if nargin == 1			% if x is not specified,
	x='n';			% norm to equal cut-off frequency ;
end

z=[];				% no zeros ;

if N == 1			% first-order system ;
	c=[1 1];
	p=-1;
	k=1;
else
	fact=[1, cumprod(1:1:2*N)];	% factorials up to 2*N ;
	binp=2 .^(0:1:N);		% binary powers ;
					% polynomial coefficients :
	c=fact(N+1:1:2*N+1)./(binp.*fact(1:1:N+1).*fact(N+1:-1:1));
end

if nargout == 1
	z=c;			% Return the polynomial coefficients
	return
end

c=c/c(N+1);			% Norm system gain to 1 for jw=0
if x == 'a' | x == 'A'		% Norm to Butterworth asyptote ;
	r=(1:1:N+1);
	g=c(1) .^((N+1-r)/N);	% c(1) is the coefficient at s^N ;
	c=c./g;			% Norm gain ;
end

p=roots(c);			% poles from polynomial coefficients ;
p=sort(p);			% sort in ascending absolute value ;
if x == 'n' | x == 'N'
	% norm p for same cuttoff frequency as the 1st-order system :
	P=p;
	y3=round(1e+4*sqrt(2)/2)/1e+4 ;		% Reference : -3 dB ;
	y=abs(freqw(P,1));			% Find attenuation at w=1 ;
% xxx=0;
	while abs( 1 - y3/y ) > 0.0001
% y > y3
		P=P*y3/y;			% Make iterative corrections ;
		% y=round(1e+4*abs(freqw(P,1)))/1e+4;
		y=abs(freqw(P,1));
% xxx=xxx+1;
	end
% xxx
% 1 - y3/y
	p=P;
end

k=real(prod(-p));		% gain.
