function q_therm(Tmin)
% Q_THERM   ... calculates transistor Pd(Tj).

%       Erik Margan, *930706*

kB = 1.38e-23 ; % [VAs/K] Boltzmann constant
qe = 1.602e-19 ; % [As] electron charge

Rthjc = 5 ; % [K/W] thermal resistance, junction to case
Rthca = 25 ; % [K/W] thermal resistance, case to ambient (TO-5, no heatsink)
Cthj = 0.05 ; % [Ws/K] junction+die thermal capacitance
Cthc = 0.65 ; % [Ws/K] case thermal capacitance

Is = 1e-14 ; % [A] saturation current in silicon
rB = 70 ; % [Ohm] bulk base resistance
beta = 275 ; % transistor dc current gain, assumed independent of temperature
alpha = 1 / ( 1 + 1 / beta ) ;

Rc = 50 ; % [Ohm] collector load
Vcc = 15 ; % [V] supply voltage
Ic = 0.1 ; % [A] initial collector current

Vo = Vcc - Ic * Rc ; % [V] initial conllector voltage
Pd = Vo * Ic ; % [W] initial power dissipation

if nargin == 0
	Tmin = 273 + 27 ;
    % room temperature 25 deg.C;
    % on a hot summer day, the room temperature can easily reach 40 deg.C;
    Tmax = 273 + 180 ;
    % 180 deg.C is the maximum temperature for silicon devices,
    % but, at this temperature, the allowable power dissipation is zero!
end

Tj = Tmin ; % initial junction temperature
Tc = Tmin ; % initial case temperature

% initial base bias voltage calculated from the required collector current at power-up
% Ib = Ic / beta 
% rE = kB * Tj / ( qe * ( beta + 1 ) * Ib ) 
% Vbb = Ib * ( rB + beta * rE )
Ie = alpha * Ic ;
Vbe = ( kB * Tj / qe ) * log( ( Ie / Is ) + 1 ) ;
Ib = Ie / ( beta + 1 ) ;
Vbb = Vbe + rB * Ib ;

% time vector
deltat = 0.2 * Rthjc * Cthj ; % increment must be smaller than the shortest time constant
tmax = 5 * Rthca * Cthc ; % max. time must be larger than the longest time constant
t = 0 : deltat : tmax ;
nt = length( t ) ;

% calculate working conditions iteratively
for k = 2:nt
    % Pd changes juction temperature
    Tj(k,:) = Tj(k-1) + deltat * ( Pd(k-1) / Cthj  - ( Tj(k-1) - Tc(k-1) ) / Rthjc ) ;
    % junction temperature changes case temperature
    Tc(k,:) = Tc(k-1) + deltat * ( ( ( Tj(k) - Tc(k-1) ) / Rthjc ) - ( ( Tc(k-1) - Tmin ) / Rthca ) ) / Cthc ;
    % Tj also changes the emitter current
    Ib = Ie / ( beta + 1 ) ;
    Vbe(k,:) = Vbb - rB * Ib ;
    Ie = Is * ( exp( Vbe(k) * qe / ( kB * Tj(k) ) ) - 1 ) ;
    % base current gives the new collector current
    Ic(k,:) = alpha * Ie ;
    Vo(k,:) = Vcc - Rc * Ic(k) ;
    % Ic changes Pd, closing the calculation loop
    Pd(:,k) = Vo(k) * Ic(k) ;
end

plot( t, Tj-273, '-r', ...
      t, Tc-273, '-m', ...
      t, Ic*1000, '-b',...
      t, Pd*1000, '-k',...
      t, Vo * 10, '-g',...
      t, Vbe * 100, '-c',...
      0, 0, '.k' )
xlabel( 'Time [s]' ) ;

