function [y,t]=ttresp(F,w,r,g)
%TTRESP	returns the system Transient RESPonses, calculated from it's
%	frequency response using FFT.
%
%	[y,t]=tresp(F,w,r,g);
%
%	
%	Example : 	[z,p]=bestap(5);
%			w=(0:1:255)/4;
%			F=freqw(z,p,w);
%			[y,t]=tresp(F,w,'s');
%	NOTE :
%	Any time-to-frequency or inverse transformation introduces an
%	oscillating error, due to the Gibbs effect. To see the amount of
%	error, work the previous example followed by :
%
%	SS=tdr(z,p,t,'s');
%	plot(t,S,t,SS), pause, plot(t,S-SS)
%
%	See FREQW, PHASE and GROUPDLY. See also TDR.

%	Erik Margan, *881101*
%	Jozef Stefan Institute, Ljubljana, Slovenia
%	Last revision :  *980929* *000229*

% ----------- Preparation of input data --------------------------
if nargin < 4
	g=0;				% no plot
end
if nargin < 3
	r='u';
end
G=(abs(F(1)));				% find system dc gain
N=max(size(F));				% find number of input points
q=max(size(w));
if q ~= N
	if q == 1
		if w < N/16
			m=w;
		else
			error('Bandwidth index w must be less than length(F)/16 !');
		end
	else
		error('Frequency vector size must be the same as F !');
	end
else
	m=find(abs(w)==1)-1;		% normalized frequency unit
	if isempty(m)
		m=0;			% could not find it.
	end
end
if m==0
	m=max(find(abs(F) >= G/sqrt(2)))-1;
end		

% ----------- Test system order ; --------------------------------
x=0;
if abs(F(1+10*m)) > 0.05
	x=1;				% probably a first-order system
elseif abs(F(1+10*m)) > 0.005
	x=2;				% probably a second-order system
end

% ----------- Form window function -------------------------------
if x > 0
	W=(1+cos(pi*(0:1:N-1)/(N-1)))/2;% right-half Hanning window
	F=W.*F;				% frequency response windowed
end

% ----------- Form normalized time-scale ; -----------------------
dt=2*pi*m/N;				% delta_t
t=dt*(0:1:N-1);

% ----------- Calculate impulse response -------------------------
y=2*real(fft(conj(F)));			% calculate inverse Fourier transform
y=y-G;					% null "offset" for low-pass systems
y=y/N;					% normalize impulse area to 1

if x == 1
	er1=dt-y(1);			% keep first point error
	y(1)=dt;			% correct first point
end
if x == 2
	y(1)=0;				
end

% ----------- Normalize the ideal impulse response ---------------
if r=='i' | r=='I'
	y=y/dt;				% ideal response
else
	if x == 1
		y=y*(1-er1/sum(y));	% correct amplitude for 'u' and 's'
	end
end
	
% ----------- Calculate step response ----------------------------
if r == 's' | r == 'S'
	if x == 1
		y = [0, y(1:N-1)];
	end
	y=cumsum(y);			% integrate to get step resp.
end

% ----------- Display results ; ------------------------------------
if g > 0
	nt = 100 ;
	plot(t(1:nt),y(1:nt))
	if r =='s' | r == 'S'
		title('Step Response')
	else
		title('Impulse Response')
	end
	xlabel('Normalized Time')
end
