function H=xyz(Z,P,re,im,h,q,tp,lb)
%XYZ	3-D system transfer-function magnitude plot over s-plane.
%
%	Call :		H=xyz(Z,P,re,im,h,q,tp,lb);
%	
%	where:		H <-- returned system magnitude over the 	s-plane
%				Z 	--> system zeros (if any)
%				P 	--> system poles
%				re --> range of the real part of the s-plane domain,
%				       defined from 0 to -abs(re)
%				im --> range of the imaginary part of the s-plane,
%				       defined from	 -im to +im
%				h  --> maximum displayed magnitude hight
%				q  --> s-domain resolution :
%				       - the real axis range is divided by q,
%				         resulting in q+1 columns of H
%				       - the imaginary axis range is divided by 2*q
%				         resulting in 2*q+1 rows of H
%				       if not specified, q default value is 50.
%				tp --> type of 3-D drawing function used :
%				         1 - mesh
%				         2 - waterfall
%				         3 - surf
%				         4 - surfl
%				       if not specified, tp default value is 1.
%				lb --> if non-zero, enables axis labeling;
%				       if not specified, lb default value is 0.
%
%	NOTE :		The shape of the graph cut along the imaginary axis
%				is the magnitude of the system frequency response :
%					M=abs(freqw(Z,P,w));  (See PATS and FREQW).
%
%	Example :		[Z,P]=buttap(2);
%					H=xyz(Z,P,-4,4,4);
 
%	Erik Margan
%	Jozef Stefan Institute, Ljubljana
%	First editing :	890112
%	Revision Matlab-IV : 980929
%	Revision Matlab-V : 000820

if nargin < 8
	lb = 0 ;
end
if nargin < 7
	tp = 1 ;
end
if nargin < 6
	q = 50 ;
end
if nargin < 5
	h = abs( re ) ;
end
if nargin < 4
	im = abs( re ) ;
end

% show only the negative side of the real axis :
re = -abs( re ) ;

% form the s-plane domain :
q = 1/q ;
d = re * ( 0 : q : 1 ) ;
w = im * ( 1 : -q : -1 ).' ;
m = length(d) ;
n = length(w) ;
H = ones( n, m ) ;

% calculate complex-frequency-response magnitude, row-by-row :
for k = 1 : m
	s = d(k) + sqrt(-1) * w ;
	if ( isempty( Z ) )
      H( :, k ) = abs( 1 ./ ( eps + pats( P, s ) ) ) ;
	else
		H( :, k ) = abs( pats( Z, s ) ./ ( eps + pats( P, s ) ) ) ;
   end
end

% limit the hights :
for k = 1 : m
	x = find( H( :, k ) > h ) ;
	if length(x) > 0
		H( x, k ) = h * ones( length(x), 1 ) ;
	end
end

% plot
% B = figure ;
A = viewmtx( 50, 30, 0 ) ;
[ Xi, Yi ] = meshgrid( d, w ) ;

if tp == 1
	mesh( Xi, Yi, H ) ;
elseif tp == 2
	waterfall( Xi', Yi', H' ) ;
elseif tp == 3
	surf( Xi, Yi, H ) ;
else
	surfl( Xi, Yi, H, [50, 30], 'light' ) ;
	colormap( 'Copper' ) ;
	caxis( [0.2, 0.9 ] ) ;
	shading interp ;
	lighting gouraud ;
end

view( A ) ;
if lb
	xlabel('sigma=Real(s)')
	ylabel('j*omega=j*Imag(s)')
	zlabel('|H(s)|')
end
