function fig601
% FIG601	Figures for CH601.WXP :
%			Complete 5th-order Butterworth system representation.
%			
%			Fig.6.4.1. Pole locations in complex plane
%			Fig.6.4.2. Magnitude over the complex plane
%			Fig.6.4.3. Complex response to imaginary frequency
%			Fig.6.4.4. Nyquist-plot magnitude/phase
%			Fig.6.4.5. Magnitude in linear scale
%			Fig.6.4.6. Bode-plot magnitude in dB vs. log(f)
%			Fig.6.4.7. Phase, modulo +/-pi
%			Fig.6.4.8. Phase unwrapped, Bode-plot
%			Fig.6.4.9. Envelope Delay
%			Fig.6.5.1. Impulse- and Step-response

% RXon, 890112
% Last revision (Matlab-V) : 001101

[z,p]=buttap(5);	% a 5th order Butterworth system zeros and poles
[row,col]=size(p);


% -------------- Fig.6.4.1. ----------------- ;
% ----- cartesian-coordinates pole loci ----- ;
figh=figure(1) ;
P=[p(:).'; zeros(size( p(:).' ) )];
P=P(:);
theta=(0:pi/180:2*pi);
fill( [ -2, 0, 0, -2 -2 ], [ -2, -2, 2, 2, -2 ], [ 0.82, 0.82, 0.82 ] ) ;
hold on
axis('manual')
plot([-1.99,1.99],[0,0],'-k',...
     [0,0],[-1.99,1.99],'-k',...
     sin(0:pi/180:2*pi),cos(0:pi/180:2*pi),'-b',...
     real(P),imag(P),'-r',...
     real(p),imag(p),'*r')
axis('square')
axis([-2,2,-2,2]);
for kx = 1 : length(p)
	text( real(p(kx))-0.3, imag(p(kx))+0.2*sign(0.1+imag(p(kx))), ['{\itp}_', num2str(kx)],'FontSize',16 ) ;
end

axh=gca;
set(axh,'XTickMode','Manual','XTickLabelMode','Manual');
set(axh,'YTickMode','Manual','YTickLabelMode','Manual');
set(axh,'XTick',[-2,-1.5,-1,-0.5,0,0.5,1,1.5,2]);
set(axh,'YTick',[-2,-1.5,-1,-0.5,0,0.5,1,1.5,2]);
set(axh,'XTickLabel',['-2.0';'-1.5';'-1.0';'-0.5';' 0.0';' 0.5';' 1.0';' 1.5';' 2.0']);
set(axh,'YTickLabel',['-2.0';'-1.5';'-1.0';'-0.5';' 0.0';' 0.5';' 1.0';' 1.5';' 2.0']);
text(1.2,0.20,'\Re\{{\its}\}','FontSize',16)
text(0.1,1.75,'\Im\{{\its}\}','FontSize',16)
% title(['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Poles'])


% -------------- Fig.6.4.2. ---------------- ;
% ----- 3-D Magnitude over the s-plane ----- ;
figh=figure(figh+1) ;
% xyz( z, p, -2, 2, 12 ) ;
% xyz( Zeros, Poles, RealRange, ImagRange, HightRange, Resolution, MeshType, LabelFalg );
xyz( z, p, -2, 2, 15, 50, 1, 0 ) ;
axis([-2,0,-2,2,0,15]);
text( -1.4, 0.6, 15, '\uparrow\infty','FontSize',16 );
axh=gca;
set(axh,'ZTick',[0,5,10,15],'ZTickLabel',[' 0';' 5';'10';'15']);
text(-1,-3,-1,'{\it\sigma} = \Re\{{\its}\}','FontSize',16)
text(0,0,-2,'{\itj\omega} = {\itj}\Im\{{\its}\}','FontSize',16)
zlabel( '{\itM}({\its}) = |{\itF}({\its})|','FontSize',16)
% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Magnitude over {\its}-Plane'] )


% -------------- Fig.6.4.3. ------------------------------- ;
% ----- 3-D Helix, complex-frequency complex response ----- ;  
figh=figure(figh+1) ;
jw=(-3:0.01:3);			% 601 frequency points, pos. and neg., linear scale
G=freqw(z,p,jw(301:601));	% 301 points of complex frequency response
G=[real(G(301:-1:2))-j*imag(G(301:-1:2)), G];	% full +/-3 freq.range

plot3(	[-3, 3], [0, 0], [0, 0], '-c',...
		[0, 0], [-1.5, 1.5], [0, 0], '-b',...
		[0, 0], [0, 0], [-1.5, 1.5], '-m',...
		jw, real(G), imag(G), '-r',...
		0, 1, 0, 'ob')
view(75,12);
axis([-3,3,-1.5,1.5,-1.5,1.5]) ;
axis('square');
axh=gca ;
set(axh,'XTick',[-3,-2,-1,0,1,2,3],'XTickLabel',['-3';'-2';'-1';' 0';' 1';' 2';' 3']);
set(axh,'YTick',[-1.5,-1,-0.5,0,0.5,1,1.5],'YTickLabel',['-1.5';' -1 ';'-0.5';'  0 ';' 0.5';'  1 ';' 1.5']);
set(axh,'ZTick',[-1.5,-1,-0.5,0,0.5,1,1.5],'ZTickLabel',['-1.5';' -1 ';'-0.5';'  0 ';' 0.5';'  1 ';' 1.5']);
text( 3, 0.2, 0.0, '{\itj\omega}','FontSize',16);
text( 0, 1.2, 0.2, '\Re\{{\itF}({\itj\omega})\}','FontSize',16 )
text( 0, 0.2, 1.5, '\Im\{{\itF}({\itj\omega})\}','FontSize',16 )
% title(['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Complex-Frequency Response'])


% -------------- Fig.6.4.4. -------------------------------------------- ;
% ----- Nyquist-plot : single-side complex-frequency complex response ----- ;
% figh=figure(figh+1) ;
figh=5;
figure(figh);
w=logspace(-3,1,601);	% 601 frequency points, equally spaced in
							% 3-decade log scale from 0.01 to 10 ;
H=freqw(z,p,w);			% 601 points of complex frequency response ;
Q=(pi/2)+(0:0.05*pi/3:pi/3);
plot([-1,1], [0,0], '-k',...
     [0,0], [-1,1], '-k',...
     0.23*sin(Q), 0.23*cos(Q), '-b',...
     1, 0, 'ob',...
     -0.5, 0.5, 'ob',...
     0, 0, 'ob',...
     real(H), imag(H), '-r')
axis('square')
axis('manual')
hold on

vector( [ 0, 0, real(H(380)), imag(H(380)) ], 'b', 1.0 ) ;
vector( [ 0.24, -0.22, 0.22, -0.24 ]/sqrt(2), 'b', 0.2 ) ;

lh = line( [ 0.45, 0.75 ], [ 0.25, 0.25 ] ) ;
set( lh, 'Color', 'k', 'LineStyle', '-', 'LineWidth', 0.5 ) ;
vector( [ 0.75, 0.25, 0.95, 0.05 ], 'k', 0.5 ) ;

lh = line( [ -0.55, -0.25 ], [ -0.25, -0.25 ] ) ;
set( lh, 'Color', 'k', 'LineStyle', '-', 'LineWidth', 0.5 ) ;
vector( [ -0.25, -0.25, -0.05, -0.05 ], 'k', 0.5 ) ;

lh = line( [ -0.25, -0.75 ], [ 0.75, 0.75 ] ) ;
set( lh, 'Color', 'k', 'LineStyle', '-', 'LineWidth', 0.5 ) ;
vector( [ -0.75, 0.75, -0.55, 0.55 ], 'k', 0.5 ) ;

% axis labels :
text(0.3, 0.07, '\Re\{{\itF}({\itj\omega})\}','FontSize',16);
text(0.05, 0.8, '\Im\{{\itF}({\itj\omega})\}','FontSize',16);

% angle label :
% fraction midline :
lh=line( [0.45, 0.95], [-0.2, -0.2] ) ;
set( lh, 'LineStyle','-','Color','k' ) ;

txh=text( 0.1, -0.07, 'x ' ) ;
% display greek 'varphi' from BCSYMB (WinEXP) font,
% istead of x , since Matlab-V does not recognize varphi :
set( txh, 'FontName', 'BCSYMB', 'FontSize', 16, 'String', setstr(58) ) ;

txh=text( 0.44, -0.2, '= {\itatan}' );
set( txh, 'HorizontalAlignment','Right','FontSize',16 ) ;
% numerator
text( 0.48, -0.12, '\Im\{{\itF}({\itj\omega})\}','FontSize',16 );
% denominator
text( 0.48, -0.28, '\Re\{{\itF}({\itj\omega})\}','FontSize',16 );

% Magnitude label :
text( 0.3, -0.5, '{\itM}({\it\omega}) = |{\itF}({\itj\omega})|','FontSize',16 );
% Frequency markers :
text( 0.5, 0.25, '{\it\omega} = 0','FontSize',16,'VerticalAlignment','Bottom');
text( -0.65, 0.75, '{\it\omega} = 1','FontSize',16,'VerticalAlignment','Bottom');
text( -0.6, -0.25, '{\it\omega} = \infty','FontSize',16,'VerticalAlignment','Bottom');

% title(['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Bode-Plot']);


% -------------- Fig.6.4.5. -------- ;
% ----- Lin-lin Magnitude plot ----- ;
figh=figure(figh+1) ;
M=abs(H);		% magnitude
ix=max(find(w<=3));	% limit the frequecy scale to 3.
% display M in lin-lin plot :
plot([w(1),w(ix)],[0.707,0.707],'-k',...
     [1,1],[0,1.19],'-k',...
     w(1:ix),M(1:ix),'-r',...
	  [1],[0.707],'ob')
axis([0, 3, 0, 1.2]);
xlabel( 'Frequency' )
ylabel( 'Magnitude' )
txh=text( 1.1, 0.74, '0.707' );
set( txh, 'FontSize',16 ) ;
% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Magnitude'] );


% -------------- Fig.6.4.6. --------------- ;
% ----- Bode-plot : log10(f)-dB(M) ----- ; 
figh=figure(figh+1) ;
w=w(301:601);		% limit the frequency to -/+1decade
H=H(301:601);
M=abs(H);		% magnitude
% display M in dB vs. log-scale w
semilogx( w, 20*log10(M),'-r', [0.7, 1.4], [-3, -3], '-b', [1, 1], [0, -10], '-b' )
axis( [ 0.1, 10, -100, 0 ] ) ;
axh=gca;
set(axh,'XTickLabel',['0.1 ';'1.0 ';'10.0']);
xlabel( 'Frequency' )
ylabel( 'Magnitude [dB]' )

text( 0.6, -6, '-3dB' )
text( 3, -40, 'Slope :' )
text( 3.5, -50, '-100dB/10f' )

% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Magnitude'] )


% -------------- Fig.6.4.7. ------------ ;
% ----- Bode-plot : phase angle ----- ;
figh=figure(figh+1) ;
phi=angle(H);		% phase response, modulo +/-pi
semilogx([min(w),max(w)],[pi,pi],'--b',...
         [min(w),max(w)],[-pi,-pi],'--b',...
         w,phi,'-r')
axh=gca;
set(axh,'XTickLabel',['0.1 ';'1.0 ';'10.0']);
xlabel( 'Frequency' )
text( max(w)/3, pi+0.25, '+\pi','FontSize',16 );
text( max(w)/3, -pi+0.25, '-\pi','FontSize',16 );
ylabel( 'Phase Angle [rad]' )
% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Phase'] )


% -------------- Fig.6.4.8. ------------------- ;
% ----- phase angle unwrapped, in degrees ----- ;
figh=figure(figh+1) ;
% unwrap the phase :
alpha=ephd(phi);
% display alpha in linear degrees
semilogx(w,180*alpha/pi,'-r')
axh=gca;
set(axh, 'YTick', [ -450, -360, -270, -180, -90, 0 ] ) ;
set(axh, 'YTickLabel', [ '-450'; '-360'; '-270'; '-180'; ' -90'; '   0' ] ) ;
set(axh, 'XTickLabel', ['0.1 ';'1.0 ';'10.0']);
xlabel( 'Frequency' )
ylabel( 'Phase Angle [\circ]' )
% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Unwrapped Phase'] )


% -------------- Fig.6.4.9. ------------- ;
% ----- group (envelope) time-delay ----- ;
figh=figure(figh+1) ;
tau=diff(alpha)./diff(w);		% time-delay
w=sqrt(w(2:1:301).*w(1:1:300));		% new w
semilogx(w,tau,'-r')			% plot
axh=gca;
set(axh,'XTickLabel',['0.1 ';'1.0 ';'10.0']);
xlabel( 'Frequency' )
ylabel( 'Time-Delay' )
% title( ['Fig. 6.4.',num2str(figh),'. :  5^{th}-order Butterworth Envelope-Delay'] )


% -------------- Fig.6.5.1. ---------------------------------- ;
% ----- time-domain response : impulse and step function ----- ;
figh=figure(figh+1) ;
w=(0:1:255)/4;		% linear frequency vector
H=freqw(z,p,w);		% complex frequency response in new w ;
[I,t]=tresp(H,w,'i');	% impulse response
[S,t]=tresp(H,w,'s');	% step response
nt=min( find( t > 20 ) ) - 1 ;	% limit time to 25*Tnorm
t=t(1:nt);
I=I(1:nt);
S=S(1:nt);
plot( t, I, '-r', t, S, '-b' )
text(8,0.15, 'Impulse-Response' )
text(8,0.85, 'Step-Response' )
xlabel( 'Time' )
ylabel( 'Amplitude' )
% title( ['Fig. 6.5.1. :  5^{th}-order Butterworth in Time Domain'] )
