function B=Fmix_z(q,m,npa,nb)
% Fmix_z    Mixed-mode (analog+digital) filter with (optional) zeros.
%           The function returns the bandwidth improvement factor.
%
%           Call :  B=Fmix_z(q,m,npa,nb);
%           where:  B --> bandwidth improvement factor
%                   q --> pole selection criterion (1 or 2)
%                         1 - selects the real and every other complex-
%                             -conjugate pair for the analog filter
%                         2 - selects the real and half of higher poles
%                   m --> number of zero pairs: 2, 1, or 0
%                 npa --> number of analog poles (default 7)
%                         the number of digital poles is npa-1
%                         the total number of poles is 2*npa-1
%                  nb --> number of bits of the ADC (default 12)

% RXon, 990719, last revision 050617, No copyright!

if nargin < 4
    nb=12;  % default number of ADC bits
end
if nargin < 3
    npa=7;  % default number of poles in the analog filter
end
if nargin < 2
    m=2;    % default number of zeros
end
if ~nargin
    q=1;    % default pole selection criterion
end

npd=npa-1;          % number of poles in the digital filter
npt=npa+npd;        % total number of poles in the mixed-mode filter
fs=20e+6;           % sampling frequency
Nf=fs/2;            % Nyquist frequency
nf=600;             % number of frequency samples
f=Nf*logspace(-1,1,nf); % frequency vector in Hz
                        % (+/-one decade about Nyquist f.)
w=2*pi*f;           % angular frequency in rad/s

A=2^nb ;            % number of ADC levels
Radc=20*log10(1/A); % ADC resolution in dB

fx=1/10^( log10(A^2 - 1)/(2*npa) );
                    % bandwidth correction factor
B=sqrt(3);          % bandwidth extension factor


[z,p]=bestap(npt,'a');  % Bessel-Thomson system with npt poles
p=p*B*fx*2*pi*Nf;       % denormalized poles

                    % selection of zeros
if m == 2
    z=2*pi*Nf*[-j;j;-2*j;2*j];
                    % a pair of zeros at Nyquist f. and a pair at sampling f.
elseif m == 1
    z=2*pi*Nf*[-2*j;2*j];
                    % a pair of zeros at sampling frequency
else
    z=[];           % no zeros
end

if q ~= 1
    if m == 2
        p=p*1.07;
        B=B*1.07
    elseif m == 1
        p=p*1.04;
        B=B*1.04;
    else
        p=p*1.001;
        B=B*1.001;
    end
    % select the real pole and every other complex-conjugate pair for analog
    k=(2:2:npt-1);
    nk=length(k);
    ka=k(2:2:nk);
    kd=k(1:2:nk-1);
    ka=[ka;ka+1];
    kd=[kd;kd+1];
    ka=ka(:);
    kd=kd(:);
    pa=p([1;ka]);
    pd=p([kd]);
else
    if m == 2
        p=p*1.081;
        B=B*1.081;
    elseif m == 1
        p=p*1.039;
        B=B*1.039;
    else
        p=p*1.01;
        B=B*1.01;
    end
    % select the real pole and the complex conjugate pole pairs with high imaginary part
    pa=p([1;(npt-npd+1:npt).']);
    pd=p([(2:npd+1).']);
end

figure(1)
% check poles and zeros to set axis scale
yp=min(abs(p));
if isempty(z)
    yy=yp;
else
    yz=max(abs(z));
    yy=max(yp,yz);
end
xs=log10(yy);
xp=floor(xs);
xs=ceil(yy*10^(-xp));
% poles and zeros in the complex plane
plot( real(pa), imag(pa), '+b', real(pd), imag(pd), 'xg', real(z), imag(z), 'or' )
axis(10^xp*[-xs,xs,-xs,xs]);
axis square;

Fo=freqw([],pa,w);  % frequency response of the analog system with no zeros
Fa=freqw(z,pa,w);   % frequency response of the analog system with zeros
Fd=freqw([],pd,w);  % frequency response of the digital system
Fs=freqw(z,p,w);    % system frequency response
figure(2)
semilogx(f,20*log10(abs(Fa)), f,20*log10(abs(Fd)), f,20*log10(abs(Fo)), f,20*log10(abs(Fs)), [f(1),f(nf)],Radc*[1,1])
axis([f(1),f(nf),-90,+10])

x1=min(find(f > Nf));   % min f index of alias spectrum
x2=max(find( 20*log10(abs(Fa)) >= Radc )); % max f of alias sp.
Aa=Fa(x1:x2);           % alias spectrum source above Radc
Ab=Fa(x1:x2);           % alias spectrum
fa=fs-f(x1:x2);         % alias frequency vector
Fr=Radc-20*log10(abs(Ab));  % min digital attenuation required
                            % for total alias suppression

Fx=freqw([],pd,2*pi*fa);    % digital f.r. fro the alias frequency vector

Fq=20*log10(abs( Ab .* Fx )); % alias suppression result

figure(3)
semilogx(f,20*log10(abs(Fa)),fa,20*log10(abs(Ab)), fa,Fr, f,20*log10(abs(Fd)), fa,Fq, [f(1),f(nf)],Radc*[1,1], [f(1),f(nf)],[0,0])
axis([f(1),f(nf),-90,+10])

nt=350;                 % number of time samples
dt=1/50;                % delta-t
t=dt*(0:1:nt)/Nf;       % time vector
Sa=atdr(z,pa,t,'s');    % analog (with zeros) step response
Xa=atdr([],pa,t,'s');   % analog (no zeros) step response
Id=atdr([],pd,t,'n');   % digital impulse response
Id=Id/(2*max(Id));      % denormalization
Ss=atdr(z,p,t,'s');     % system step response
figure(4)
plot(t,Sa,t,Id,t,Xa,t,Ss)
