function gptline(h,w,t,l,k,epsr)
% GPTLINE       Ground-Plane Transmission LINE design aid. 
%               Micro-strip configuration is assumed.
%       Call :
%               gptline(h,w,t,l,k,epsr);
%       where:  
%               h <-- track hight (above ground)             [mm]
%               w <-- track width                            [mm]
%               t <-- trach thickness                        [mm]
%               l <-- track length                           [mm]
%               k <-- coupling factor ( 1 < k < 2 )
%            epsr <-- relative dielectric constant ( 1 < epsr < 20 )

%               s <-- track spacing                          [mm] ?????

%       Erik Margan, *940420*

if nargin == 0
        h=1.0;
        w=1.0;
        t=0.038; % 0.076;
        l=150.0;
        s=2.54;
        k=1;
        epsr=4.7;
end

disp(['    h = ', num2str(h), ' [mm]']);
disp(['    w = ', num2str(w), ' [mm]']);
disp(['    t = ', num2str(t), ' [mm]']);
disp(['    l = ', num2str(l), ' [mm]']);
% disp(['    s = ', num2str(s), ' [mm]']);
disp(['    k = ', num2str(k)]);
disp([' epsr = ', num2str(epsr)]);
disp(' ');

h=1e-3*h;
w=1e-3*w;
t=1e-3*t;
l=1e-3*l;
s=1e-3*s;

if w/h < 0.1 | w/h > 3
        disp(['WARNING !     w/h = ', num2str(w/h)]);
        disp('This is outside the range (0.1 ... 3.0), where accuracy is +/-5% .');
end

if epsr < 1
        error('epsr must be >= 1 !');
elseif epsr > 15
        disp(['WARNING !     epsr = ', num2str(epsr)]);
        disp('This is outside the range (1 ... 15), where accuracy is +/-5% .');
end

if k < 1 | k > 2
        error('Coupling coefficient must be 1 < k < 2 ; typ.: k=1.95 !');
end

% constants :
mur=1;
mu0=pi*4e-7;            % [Vs/Am]
eps0=8.85e-12;          % [As/Vm]

% approx. impedance of the single track element cross-section :
% zo=(377/sqrt(epsr))*(d/w);
zo=(87/sqrt(epsr+1.41))*log(5.98*h/(0.8*w+t));

% approx. line capacitance and inductance of a single track element :
co=(100/30.48)*eps0*epsr*w/(2*h);               % [F/m]
lo=co*zo^2;                                     % [H/m]
td=(100/30.48)*1.017*sqrt(0.475*epsr+0.67);     % [ns/m]

disp('   zo [ohm]    co [pF/m]  lo [nH/m]  td [ns/m]  ');
disp('------------------------------------------------');
disp([zo, co*1e+12, lo*1e+9, td]);
disp(' ');

Co=co*l;
Lo=lo*l;
Zo=zo;
tx=td*l;

disp('   Zo [ohm]    Co [pF]    Lo [nH]    tx [ns]    ');
disp('------------------------------------------------');
disp([Zo, Co*1e+12, Lo*1e+9, tx]);
disp(' ');

% propagation speed :
vp=1/sqrt(lo*co);
% total time delay :
taud=l/vp;
c=1/sqrt(mu0*eps0);

disp(' c [mm/ns]  v [mm/ns]  taud [ns] ');
disp('---------------------------------');
disp([c*1e-6,vp*1e-6, taud*1e+9]);
