function gptline(h,w,t,l,epsr)
% GPTLINE       Ground-Plane Transmission LINE design aid. 
%               Micro-strip configuration is assumed.
%       Call :
%               gptline(h,w,t,l,epsr);
%       where:  
%               h <-- track hight (above ground)             [mm]
%               w <-- track width                            [mm]
%               t <-- trach thickness                        [mm]
%               l <-- track length                           [mm]
%            epsr <-- relative dielectric constant (1<epsr<15)

%       Erik Margan, *940420*

if nargin == 0
        h=1.0;
        w=1.0;
        t=0.038; % 0.076;
        l=150.0;
        epsr=4.7;
end

disp(['            h = ', num2str(h), ' [mm]']);
disp(['            w = ', num2str(w), ' [mm]']);
disp(['            t = ', num2str(t), ' [mm]']);
disp(['            l = ', num2str(l), ' [mm]']);
disp(['         epsr = ', num2str(epsr)]);
disp(' ');

h=1e-3*h;
w=1e-3*w;
t=1e-3*t;
l=1e-3*l;
s=1e-3*s;

if w/h < 0.1 | w/h > 3
        disp(['WARNING !     w/h = ', num2str(w/h)]);
        disp('Value outside the +/-5% accuracy range (0.1 ... 3.0).');
end

if epsr < 1
        error('epsr must be >= 1 !');
elseif epsr > 15
        disp(['WARNING !     epsr = ', num2str(epsr)]);
        disp('Value outside the +/-5% accuracy range (1 ... 15).');
end

% constants :
mur=1;
mu0=pi*4e-7;            % [Vs/Am]
eps0=8.85e-12;          % [As/Vm]

% approx. impedance of the single track element cross-section :
% zo=(377/sqrt(epsr))*(d/w);
zo=(87/sqrt(epsr+1.41))*log(5.98*h/(0.8*w+t));

% approx. line capacitance and inductance of a single track element :
co=(100/30.48)*eps0*epsr*w/(2*h);               % [F/m]
lo=co*zo^2;                                     % [H/m]
td=(100/30.48)*1.017*sqrt(0.475*epsr+0.67);     % [ns/m]

disp('   zo [ohm]    co [pF/m]  lo [nH/m]  td [ns/m]  ');
disp('------------------------------------------------');
disp([zo, co*1e+12, lo*1e+9, td]);
disp(' ');

Co=co*l;
Lo=lo*l;
Zo=zo;
tx=td*l;        % [ns]

disp('   Zo [ohm]    Co [pF]    Lo [nH]    tx [ns]    ');
disp('------------------------------------------------');
disp([Zo, Co*1e+12, Lo*1e+9, tx]);
disp(' ');

% propagation speed :
% vp=1/sqrt(lo*co);
vp=1e+9*l/tx;
% total time delay :
taud=l/vp;
% light propagation speed :
c=1/sqrt(mu0*eps0);

disp(' c [mm/ns]  v [mm/ns]  taud [ns]  ');
disp('----------------------------------');
disp([c*1e-6,vp*1e-6, taud*1e+9]);
