function tau=gdly(Z,P,w,f0)
% GDLY ... returns the group (envelopr) time delay from system
%	   zeros Z and poles P for the frequency interval w.
%
%	Use :    tau=gdly(Z,P,w);
%
%	If F=freqw(Z,P,w) is the complex frequency response, then
%	phi=ephd(angle(F)) gives the phase angle unwrapped and
%	tau=diff(phi) ./ diff(w)  is the group (envelope) delay
%	                          which is the longer way to it.
%	Also, in this way, length(tau) will be length(w)-1, due to
%	differetiation, and must be comensated for before plotting
%	against w. Instead, GDLY returns the correct length.
%
%	Although the group delay is defined as a (positive!)
%	time lag, by which the system response lags the input,
%	this routine returns a negative value, since this reflects
%	the sense of phase rotation with frequency (when differetiated).
%	
%	See FREQW, PATS, ABS, ANGLE, PHASE.

%	Erik Margan, 890414
%	Institute Jozef Stefan, Ljubljana, Slovenia
%	Last revision :  980925

if nargin == 2
	w = P ;
	P = Z ;
	Z = [];		% system has poles only.
end

if any( real( P ) > 0 )
	disp( 'WARNING :  This is not a Hurwitz-type system !' )
end

if any( w == i*imag(w) )
	w = imag(w) ;	% w must be real.
end

if nargin == 4
	w = 2*pi*f0*w ;	% denormalize to f0.
end

n = length( P ) ;
m = length( Z ) ;

[ r, c ] = size( w ) ;
if c == 1
	w = w(:).' ;	% make it a row vector.
end

tau( 1, : ) = real( P(1) ) ./ ( real( P(1) )^2 + ( w - imag( P(1) ) ).^2 ) ;
if n > 1
	for k = 2 : n
		tau( 2, : ) = real( P(k) ) ./ ( real( P(k) )^2 + ( w - imag( P(k) ) ).^2 ) ;
		tau( 1, : ) = sum( tau ) ;
	end
end

if m > 0
	for k = 1 : m
		tau( 2, : ) = - real( Z(k) ) ./ ( real( Z(k) )^2 + ( w - imag( Z(k) ) ).^2 ) ;
		tau( 1, : ) = sum( tau ) ;
	end
end
if ( n > 1 ) | ( m > 0 )
	tau( 2, : ) = [] ;
end

if c == 1
	tau = tau(:) ;
end
