function dtline(h,w,t,l,s,k,epsr)
% DTLINE ...Differentially driven Transmission LINE design aid. 
%           Differential meandre form is assumed.
%       Call :
%               dtline(h,w,t,l,s,k,epsr);
%       where:  
%               h <-- hight between upper and lower track       [mm]
%               w <-- track width                               [mm]
%               t <-- trach thickness                           [mm]
%               l <-- track segment length (system width)       [mm]
%               s <-- track spacing                             [mm]
%               n <-- number of track segments (n*s = system length)
%               k <-- coupling factor                  (1 <= k <= 2)
%            epsr <-- relative dielectric constant (1 <= epsr <= 50)

%       Erik Margan, *940420*

if nargin == 0
        h=1.0;
        w=0.4;
        t=0.04;
        l=6.0;
        s=1.0;
        n=100;
        k=1.95;
        epsr=15;
end

disp(['        h = ', num2str(h), ' [mm]']);
disp(['        w = ', num2str(w), ' [mm]']);
disp(['        t = ', num2str(t), ' [mm]']);
disp(['        l = ', num2str(l), ' [mm]']);
disp(['        s = ', num2str(s), ' [mm]']);
disp(['        n = ', num2str(n)]);
disp(['        k = ', num2str(k)]);
disp(['     epsr = ', num2str(epsr)]);

h=1e-3*h;
w=1e-3*w;
t=1e-3*t;
l=1e-3*l;
s=1e-3*s;

if k < 1 | k > 2
        error('Coupling coefficient must be 1 < k < 2 ; typ.: k=1.95 !');
end
if w < 3*t
        error('Track width w must be > 3*t !');
end
if h < 10*t
        error('Track distance h must be > 10*t !');
end
if s > 3*w
        error('Track spacing s must be < 3*w for coupling k > 1 !');
end

% constants :
mu0=pi*4e-7;            % [Vs/Am]
mur=1;
eps0=8.85e-12;          % [As/Vm]

% approx. impedance of the single track element cross-section :
% zo=(377/sqrt(epsr))*(d/w);
zo=(87/sqrt(epsr+1.41))*log(0.5*5.98*h/(0.8*w+t));

% approx. line capacitance and inductance of a single track element :
co=eps0*epsr*w*l/h;
lo=co*zo^2;

disp(' ');
disp('   zo [ohm]    co [pF]    lo [nH]  ');
disp('-----------------------------------');
disp([zo, co*1e+12, lo*1e+9]);
disp(' ');

% light propagation speed :
c=1/sqrt(mu0*eps0);
% specific time delay :
tds=k*1.017*(sqrt(0.475*epsr+0.67))/0.3048;     % [ns/m]
% propagation speed :
vp=1/tds;       % [m/ns]
% segment time delay :
td=tds*l;       % [ns]
% total time delay :
taud=n*td;      % [ns]

% total line length :
% n*l
% system length :
% n*s

disp(' ');
disp(' c [mm/ns]  v [mm/ns]  taud [ns]  tll [mm]  sl [mm]  ');
disp('-----------------------------------------------------');
disp([c*1e-6, vp*1000, taud, n*l*1000, n*s*1000]);
