function [y,t]=tresp(F,w,r,g,h)
%TRESP   Transient RESPonse, using Fast-Fourier-Transform algorhythm.
%
%	Call : [y,t]=tresp(F,w,r,g,h);
%
%	input arguments:
%		F is a 2^N length vector of complex-frequency response
%		w can be the related frequency vector of F, or it
%		  can be the normalized frequency unit index, or it
%		  can be zero and the n.f.u. index is found from F ;
%		r is a character, selects the response type returned in y:
%		- 'i' is the ideal impulse response
%		- 'u' is the unity-area impulse response (the default)
%		- 's' is the step response
%		g is optional; if present and nonzero, plot the response.
%		h is optional; if 1, force using windowed f.r.,
%			       if 0, force using unwindowed f.r.,
%			       if -1, use windowed f.r. only for order<3 systems
%	output results:
%		y is the selected system response
%		t is the normalized time-scale vector.

%	Erik Margan, 880414, Last revision 000316

% ----------- Preparation and check of input data ; ----------------
if nargin < 5
	h=-1;
end
if nargin < 4
	g=0;
end
if nargin < 3
	r='u';			% default response selector ;
end
if nargin < 2
	w=0;
end
G=abs(F(1));			% find system d.c. gain ;
N=length(F);			% find number of input points ;
v=length(w);
if v == 1
	m = w ;			% w is the normalized frequency unit or zero;
elseif v == N
	m=find(abs(w-1)==min(abs(w-1)))-1;	% find the normalized frequency unit;
	if isempty(m)
		m = 0 ;		% could not find it, try from half-power bandwidth;
	end ;
else
	error('F and w must be of equal length !');
end
if m == 0
	m=max(find(abs(F)>=G/sqrt(2)))-1;
end

% check magn.slope between 2 and 3 octaves above cut-off;
M=abs(diff(20*log10(abs(F(1+4*m*[1,2])))));
x=3;				% system is third-order or higher;
if M < 15
	x=2;			% probably a second-order system;
end
if M < 9
	x=1;			% probably a first-order system;
end

% ----------- Normalize the time-scale ; ---------------------------
A=2*pi*m;			% amplitude denormalization factor;
dt=A/N;				% calculate delta_t
if v == N
	dt=dt/w(m+1);		% correct for the actual frequency unit;
end
t=dt*(0:1:N-1);			% form normalized time scale ;

% ----------- Form window function ; -------------------------------
if ( h > 0) | ( ( x < 3 ) & ( h ~= 0 ) )
	% W=(1+cos(pi*(0:1:N-1)/(N-1)))/2; % right-half Hanning window ;
	W=0.54-0.46*cos(2*pi*(N+1:2*N)/(2*N)); % right-half Hamming window ;
	F=W.*F;			% frequency response windowed ;
end

% ----------- Calculate impulse response ; -------------------------
y=2*real(fft(conj(F)))-G;	% calculate iFFT and null d.c. offset ;
if x == 1
	er1=A*G-y(1);
	y(1)=A*G;
	if ~h
		y(1)=0.925*y(1);
	end
end
if r == 'u' | r == 'U' | r == 's' | r == 'S'
	y=y/N;			% normalize area to M ;
end

% ----------- Calculate step response ; ----------------------------
if r == 's' | r == 'S'
	if x == 1
		y=y*(1/(1+er1/N));	% correct first-point error ;
	end
	y=cumsum([0, y]);		% integrate to get the step resp. ;
	if x > 1
		y=(y(1:N)+y(2:N+1))/2;	% compensate the half-sample-time delay ;
		y(1)=0;
	else
		y=y(1:N);
	end
end

% ----------- Normalize Amplitude to Ideal ; -----------------------
if r == 'i' | r == 'I'
	y=y/A;			% normalize impulse amplitude ;
end

% ----------- Display results ; ------------------------------------
if g
	xt = min( find( ( t - 20 ) > 0 ) ) - 1 ;
	
	plot( t(1:xt), y(1:xt), '-r' ), xlabel('Time')
	if r == 'i' | r == 'I' | r == 'u' | r == 'U'
		title(['Impulse-response, ', num2str(xt), ' samples shown'])
	else
		title(['Step-response, ', num2str(xt), ' samples shown'])
	end
end
